"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "adminRoutes", {
    enumerable: true,
    get: function() {
        return adminRoutes;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _lodash = require("lodash");
const _middleware = require("@tamanu/shared/permissions/middleware");
const _errors = require("@tamanu/shared/errors");
const _constants = require("@tamanu/constants");
const _settings = require("@tamanu/settings");
const _importerEndpoint = require("./importerEndpoint");
const _programImporter = require("./programImporter");
const _referenceDataImporter = require("./referenceDataImporter");
const _surveyResponsesImporter = require("./surveyResponsesImporter");
const _exporter = require("./exporter");
const _patientMerge = require("./patientMerge");
const _sync = require("./sync");
const _fhirJobStats = require("./fhirJobStats");
const _reportRoutes = require("./reports/reportRoutes");
const _template = require("./template");
const _asset = require("./asset");
const _translation = require("./translation");
const _exportProgram = require("./programExporter/exportProgram");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _insurerPaymentImporter = require("./invoice/insurerPaymentImporter");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const adminRoutes = _express.default.Router();
adminRoutes.use(_middleware.ensurePermissionCheck);
adminRoutes.use('/reports', _reportRoutes.reportsRouter);
adminRoutes.use('/translation', _translation.translationRouter);
adminRoutes.post('/mergePatient', _patientMerge.mergePatientHandler);
// A temporary lookup-patient-by-displayId endpoint, just to
// support patient merge because the patient search functionality is only
// available on Facility and there was some time pressure to get it out the door.
// This should be replaced by the full-fledged patient search once some
// more consideration has been put into how that functionality should best
// be shared between the server modules.
adminRoutes.get('/lookup/patient/:displayId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'Patient');
    const { Patient } = req.store.models;
    const { displayId } = req.params;
    const patient = await Patient.findOne({
        where: {
            displayId
        },
        include: [
            'village'
        ]
    });
    if (!patient) throw new _errors.NotFoundError(`Could not find patient with display ID ${displayId}.`);
    res.send(patient);
}));
adminRoutes.post('/import/referenceData', (0, _importerEndpoint.createDataImporterEndpoint)(_referenceDataImporter.referenceDataImporter));
adminRoutes.post('/import/program', (0, _importerEndpoint.createDataImporterEndpoint)(_programImporter.programImporter));
adminRoutes.post('/import/surveyResponses', (0, _importerEndpoint.createDataImporterEndpoint)(_surveyResponsesImporter.surveyResponsesImporter));
adminRoutes.post('/import/insurerPayments', (0, _importerEndpoint.createDataImporterEndpoint)(_insurerPaymentImporter.insurerPaymentImporter));
adminRoutes.get('/export/referenceData', (0, _expressasynchandler.default)(async (req, res)=>{
    const { store, query } = req;
    const { includedDataTypes = {} } = query;
    for (const dataType of Object.values(includedDataTypes)){
        // When it is ReferenceData, check if user has permission to list ReferenceData
        if ([
            'diagnosis',
            ..._constants.REFERENCE_TYPE_VALUES
        ].includes(dataType)) {
            req.checkPermission('list', 'ReferenceData');
            continue;
        }
        // Otherwise, if it is other types (eg: patient, lab_test_types,... ones that have their own models)
        // check the permission against the models
        const nonReferenceDataModelName = (0, _lodash.upperFirst)(dataType);
        req.checkPermission('list', nonReferenceDataModelName);
    }
    const filename = await (0, _exporter.exporter)(store, query.includedDataTypes);
    res.download(filename);
}));
adminRoutes.get('/export/program/:programId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'Program');
    const { store } = req;
    const { programId } = req.params;
    const filename = await (0, _exportProgram.exportProgram)(store, programId);
    res.download(filename);
}));
adminRoutes.get('/programs', (0, _crudHelpers.simpleGetList)('Program'));
adminRoutes.get('/sync/lastCompleted', _sync.syncLastCompleted);
adminRoutes.get('/fhir/jobStats', _fhirJobStats.fhirJobStats);
adminRoutes.use('/template', _template.templateRoutes);
adminRoutes.use('/asset', _asset.assetRoutes);
// These settings endpoints are setup for viewing and saving the settings in the JSON editor in the admin panel
adminRoutes.get('/settings', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('read', 'Setting');
    const { Setting } = req.store.models;
    const data = await Setting.get('', req.query.facilityId, req.query.scope);
    res.send(data);
}));
adminRoutes.put('/settings', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Setting');
    const { Setting } = req.store.models;
    await Setting.set('', req.body.settings, req.body.scope, req.body.facilityId);
    res.json({
        code: 200
    });
}));
adminRoutes.delete('/settings/cache', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('manage', 'all');
    _settings.settingsCache.reset();
    res.status(204).send();
}));
adminRoutes.get('/facilities', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'Facility');
    const { Facility } = req.store.models;
    const data = await Facility.findAll({
        attributes: [
            'id',
            'name'
        ]
    });
    res.send(data);
}));

//# sourceMappingURL=index.js.map