"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    provision: function() {
        return provision;
    },
    provisionCommand: function() {
        return provisionCommand;
    }
});
const _path = require("path");
const _commander = require("commander");
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _database = require("../database");
const _integrations = require("../integrations");
const _loadSettingFile = require("../utils/loadSettingFile");
const _referenceDataImporter = require("../admin/referenceDataImporter");
const _utils = require("../auth/utils");
const _programImporter = require("../admin/programImporter/programImporter");
async function provision(provisioningFile, { skipIfNotNeeded }) {
    const store = await (0, _database.initDatabase)({
        testMode: false
    });
    const userCount = await store.models.User.count();
    if (userCount > 0) {
        if (skipIfNotNeeded) {
            _logging.log.info(`Found ${userCount} users already in the database, but expecting to, not provisioning`);
            return;
        }
        throw new Error(`Found ${userCount} users already in the database, aborting provision`);
    }
    (0, _integrations.checkIntegrationsConfig)();
    const { users = {}, facilities = {}, programs = [], referenceData = [], settings: globalSettings = {} } = await (0, _loadSettingFile.loadSettingFile)(provisioningFile);
    /// //////////////
    /// REFERENCE DATA
    const errors = [];
    const stats = [];
    const importerOptions = {
        errors,
        models: store.models,
        stats,
        includedDataTypes: [
            ..._constants.GENERAL_IMPORTABLE_DATA_TYPES,
            ..._constants.PERMISSION_IMPORTABLE_DATA_TYPES
        ],
        checkPermission: ()=>true
    };
    for (const { file: referenceDataFile = null, url: referenceDataUrl = null, ...rest } of referenceData ?? []){
        if (!referenceDataFile && !referenceDataUrl) {
            throw new Error(`Unknown reference data import with keys ${Object.keys(rest).join(', ')}`);
        }
        if (referenceDataFile) {
            const realpath = (0, _path.resolve)(provisioningFile, referenceDataFile);
            _logging.log.info('Importing reference data file', {
                file: realpath
            });
            await (0, _referenceDataImporter.referenceDataImporter)({
                file: realpath,
                ...importerOptions
            });
        } else if (referenceDataUrl) {
            _logging.log.info('Downloading reference data file', {
                url: referenceDataUrl
            });
            const file = await fetch(referenceDataUrl);
            const data = Buffer.from(await (await file.blob()).arrayBuffer());
            _logging.log.info('Importing reference data', {
                size: data.byteLength
            });
            await (0, _referenceDataImporter.referenceDataImporter)({
                data,
                file: referenceDataUrl,
                ...importerOptions
            });
        }
    }
    if (errors.length) {
        for (const error of errors){
            _logging.log.error(error);
        }
        throw new Error(`Encountered ${errors.length} errors during provisioning`);
    }
    _logging.log.info('Imported reference data successfully', stats);
    /// //////////
    /// FACILITIES
    for (const [id, value] of Object.entries(facilities)){
        const fields = {
            ...value
        };
        delete fields.user;
        delete fields.password;
        delete fields.settings;
        const facility = await store.models.Facility.findByPk(id);
        if (facility) {
            _logging.log.info('Updating facility', {
                id
            });
            await facility.update(fields);
        } else {
            var _fields, _fields1;
            _logging.log.info('Creating facility', {
                id
            });
            (_fields = fields).name || (_fields.name = id);
            (_fields1 = fields).code || (_fields1.code = id);
            await store.models.Facility.create({
                id,
                ...fields
            });
        }
    }
    /// ////////
    /// SETTINGS
    for (const [key, value] of Object.entries(globalSettings)){
        _logging.log.info('Installing global setting', {
            key
        });
        await store.models.Setting.set(key, value, _constants.SETTINGS_SCOPES.GLOBAL);
    }
    for (const [id, { settings = {} }] of Object.entries(facilities)){
        for (const [key, value] of Object.entries(settings)){
            _logging.log.info('Installing facility setting', {
                key,
                facility: id
            });
            await store.models.Setting.set(key, value, _constants.SETTINGS_SCOPES.FACILITY, id);
        }
    }
    /// /////
    /// USERS
    const allUsers = [
        ...Object.entries(users),
        ...Object.entries(facilities).map(([id, { user, password }])=>user && password && [
                user,
                {
                    displayName: `System: ${id} sync`,
                    password
                }
            ]).filter(Boolean)
    ];
    for (const [email, { role = 'admin', password, ...fields }] of allUsers){
        let realPassword = password;
        if (!realPassword) {
            realPassword = (0, _utils.getRandomBase64String)(16);
            // eslint-disable-next-line no-console
            console.log(`NEW PASSWORD for ${email}: ${realPassword}`);
        }
        const user = await store.models.User.findOne({
            where: {
                email
            }
        });
        if (user) {
            _logging.log.info('Updating user', {
                email
            });
            user.set({
                role,
                ...fields
            });
            user.setPassword(realPassword);
            await user.save();
        } else {
            _logging.log.info('Creating user', {
                email
            });
            await store.models.User.create({
                email,
                role,
                password: realPassword,
                ...fields
            });
        }
    }
    _logging.log.info('Creating system user');
    await store.models.User.create({
        id: _constants.SYSTEM_USER_UUID,
        email: 'system@tamanu.io',
        role: 'system',
        displayName: 'System'
    });
    /// ////////
    /// PROGRAMS
    const programOptions = {
        errors,
        models: store.models,
        stats,
        checkPermission: ()=>true
    };
    for (const { file: programFile = null, url: programUrl = null, ...rest } of programs){
        if (!programFile && !programUrl) {
            throw new Error(`Unknown program import with keys ${Object.keys(rest).join(', ')}`);
        }
        if (programFile) {
            const realpath = (0, _path.resolve)(provisioningFile, programFile);
            _logging.log.info('Importing program file', {
                file: realpath
            });
            await (0, _programImporter.programImporter)({
                file: realpath,
                ...programOptions
            });
        } else if (programUrl) {
            _logging.log.info('Downloading program file', {
                url: programUrl
            });
            const file = await fetch(programUrl);
            const data = Buffer.from(await (await file.blob()).arrayBuffer());
            _logging.log.info('Importing program', {
                size: data.byteLength
            });
            await (0, _programImporter.programImporter)({
                data,
                file: programUrl,
                ...programOptions
            });
        }
    }
    if (errors.length) {
        for (const error of errors){
            _logging.log.error(error);
        }
        throw new Error(`Encountered ${errors.length} errors during provisioning`);
    }
    _logging.log.info('Imported programs successfully', stats);
    _logging.log.info('Done.');
}
const provisionCommand = new _commander.Command('provision').description('Set up initial data. See https://beyond-essential.slab.com/posts/tamanu-provisioning-file-h1urgi86 for details or /docs/provisioning/example.json5 for a sample file.').argument('<file>', 'Path to the provisioning file').option('--skip-if-not-needed', 'If there are already users in the database, exit(0) instead of aborting').action(provision);

//# sourceMappingURL=provision.js.map