"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    makeCovidCertificate: function() {
        return makeCovidCertificate;
    },
    makeCovidVaccineCertificate: function() {
        return makeCovidVaccineCertificate;
    },
    makeVaccineCertificate: function() {
        return makeVaccineCertificate;
    }
});
const _react = /*#__PURE__*/ _interop_require_default(require("react"));
const _renderer = /*#__PURE__*/ _interop_require_default(require("@react-pdf/renderer"));
const _path = /*#__PURE__*/ _interop_require_default(require("path"));
const _qrcode = /*#__PURE__*/ _interop_require_default(require("qrcode"));
const _lodash = require("lodash");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _utils = require("@tamanu/shared/utils");
const _patientCertificates = require("@tamanu/shared/utils/patientCertificates");
const _localisation = require("../localisation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
async function getCertificateAssets(models, footerAssetName) {
    const footerAsset = await models.Asset.findOne({
        raw: true,
        where: {
            name: footerAssetName
        }
    });
    const footerAssetData = footerAsset?.data;
    const [logo, watermark, signingImage] = (await Promise.all([
        _constants.ASSET_NAMES.LETTERHEAD_LOGO,
        _constants.ASSET_NAMES.VACCINE_CERTIFICATE_WATERMARK,
        ...footerAsset?.data ? [] : [
            _constants.ASSET_FALLBACK_NAMES[footerAssetName] || _constants.ASSET_NAMES.CERTIFICATE_BOTTOM_HALF_IMG
        ]
    ].map((name)=>name && models.Asset.findOne({
            raw: true,
            where: {
                name
            }
        })))).map((record)=>record?.data); // avoids having to do ?.data in the prop later
    return {
        logo,
        signingImage: footerAssetData || signingImage,
        watermark
    };
}
async function renderPdf(element, fileName) {
    const folder = await (0, _utils.tmpdir)();
    const filePath = _path.default.join(folder, fileName);
    await _renderer.default.render(element, filePath);
    return {
        status: 'success',
        filePath
    };
}
async function getPatientVaccines(models, patient) {
    const { data: vaccines } = await patient.getAdministeredVaccines({
        order: [
            [
                'date',
                'ASC'
            ]
        ],
        includeNotGiven: false
    });
    const vaccineData = vaccines.filter((vaccine)=>!vaccine.scheduledVaccine.hideFromCertificate);
    const certifiableVaccines = vaccineData.filter((vaccine)=>vaccine.certifiable);
    const additionalData = await models.PatientAdditionalData.findOne({
        where: {
            patientId: patient.id
        },
        include: models.PatientAdditionalData.getFullReferenceAssociations()
    });
    const village = await models.ReferenceData.findByPk(patient.villageId, {});
    const patientData = {
        ...patient.dataValues,
        village,
        additionalData: additionalData?.dataValues
    };
    return {
        certifiableVaccines,
        vaccines: vaccineData,
        patientData
    };
}
const makeCovidVaccineCertificate = async (patient, printedBy, printedDate, models, uvci, qrData = null, language)=>{
    const localisation = await (0, _localisation.getLocalisation)();
    const getLocalisationData = (key)=>(0, _lodash.get)(localisation, key);
    const fileName = `covid-vaccine-certificate-${patient.id}.pdf`;
    const { logo, signingImage, watermark } = await getCertificateAssets(models, _constants.ASSET_NAMES.COVID_VACCINATION_CERTIFICATE_FOOTER);
    const { certifiableVaccines, patientData } = await getPatientVaccines(models, patient);
    const vds = qrData ? await _qrcode.default.toDataURL(qrData) : null;
    return renderPdf(/*#__PURE__*/ _react.default.createElement(_utils.CovidVaccineCertificate, {
        patient: patientData,
        printedBy: printedBy,
        printedDate: printedDate,
        uvci: uvci,
        vaccinations: certifiableVaccines,
        signingSrc: signingImage,
        watermarkSrc: watermark,
        logoSrc: logo,
        vdsSrc: vds,
        getLocalisation: getLocalisationData,
        language: language
    }), fileName);
};
const makeVaccineCertificate = async (patient, printedBy, printedDate, facilityName, models, language, translations)=>{
    const localisation = await (0, _localisation.getLocalisation)();
    const fileName = `vaccine-certificate-${patient.id}.pdf`;
    const { logo, signingImage, watermark } = await getCertificateAssets(models, _constants.ASSET_NAMES.VACCINATION_CERTIFICATE_FOOTER);
    const { vaccines, patientData } = await getPatientVaccines(models, patient);
    return renderPdf(/*#__PURE__*/ _react.default.createElement(_utils.VaccineCertificate, {
        patient: patientData,
        printedBy: printedBy,
        printedDate: printedDate,
        vaccinations: vaccines,
        facilityName: facilityName,
        signingSrc: signingImage,
        watermarkSrc: watermark,
        logoSrc: logo,
        translations: translations,
        localisation: localisation,
        language: language
    }), fileName);
};
const makeCovidCertificate = async (certType, patient, printedBy, models, vdsData = null, language)=>{
    const localisation = await (0, _localisation.getLocalisation)();
    const getLocalisationData = (key)=>(0, _lodash.get)(localisation, key);
    const fileName = `covid-${certType}-certificate-${patient.id}.pdf`;
    const footerAssetName = certType === _patientCertificates.CertificateTypes.test ? _constants.ASSET_NAMES.COVID_TEST_CERTIFICATE_FOOTER : _constants.ASSET_NAMES.COVID_CLEARANCE_CERTIFICATE_FOOTER;
    const { logo, signingImage, watermark } = await getCertificateAssets(models, footerAssetName);
    const vds = vdsData ? await _qrcode.default.toDataURL(vdsData) : null;
    const additionalData = await models.PatientAdditionalData.findOne({
        where: {
            patientId: patient.id
        },
        include: models.PatientAdditionalData.getFullReferenceAssociations()
    });
    const passportFromSurveyResponse = await (0, _utils.getPatientSurveyResponseAnswer)(models, patient.id, _config.default?.questionCodeIds?.passport);
    const nationalityId = await (0, _utils.getPatientSurveyResponseAnswer)(models, patient.id, _config.default?.questionCodeIds?.nationalityId);
    const nationalityRecord = await models.ReferenceData.findByPk(nationalityId);
    const nationalityFromSurveyResponse = nationalityRecord?.dataValues?.name;
    const patientData = {
        ...patient.dataValues,
        additionalData: {
            ...additionalData?.dataValues,
            passport: additionalData?.dataValues?.passport || passportFromSurveyResponse,
            nationality: {
                name: additionalData?.dataValues?.nationality?.name || nationalityFromSurveyResponse
            }
        }
    };
    const labs = certType === _patientCertificates.CertificateTypes.test ? await patient.getCovidLabTests() : await patient.getCovidClearanceLabTests();
    return renderPdf(/*#__PURE__*/ _react.default.createElement(_patientCertificates.CovidLabCertificate, {
        patient: patientData,
        labs: labs,
        signingSrc: signingImage,
        watermarkSrc: watermark,
        logoSrc: logo,
        printedBy: printedBy,
        vdsSrc: vds,
        getLocalisation: getLocalisationData,
        certType: certType,
        language: language
    }), fileName);
};

//# sourceMappingURL=makePatientCertificate.js.map