"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "LabRequest", {
    enumerable: true,
    get: function() {
        return LabRequest;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("./buildEncounterLinkedSyncFilter");
const _dateTimeTypes = require("./dateTimeTypes");
const _dateTime = require("../utils/dateTime");
const _generateDisplayId = require("../utils/generateDisplayId");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
let LabRequest = class LabRequest extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            sampleTime: (0, _dateTimeTypes.dateTimeType)('sampleTime', {
                allowNull: true
            }),
            requestedDate: (0, _dateTimeTypes.dateTimeType)('requestedDate', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            specimenAttached: {
                type: _sequelize.Sequelize.BOOLEAN,
                defaultValue: false
            },
            urgent: {
                type: _sequelize.Sequelize.BOOLEAN,
                defaultValue: false
            },
            status: {
                type: _sequelize.Sequelize.STRING,
                defaultValue: _constants.LAB_REQUEST_STATUSES.RECEPTION_PENDING
            },
            reasonForCancellation: {
                type: _sequelize.Sequelize.STRING
            },
            senaiteId: {
                type: _sequelize.Sequelize.STRING,
                allowNull: true
            },
            sampleId: {
                type: _sequelize.Sequelize.STRING,
                allowNull: true
            },
            displayId: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false,
                defaultValue () {
                    return (0, _generateDisplayId.generateDisplayId)();
                }
            },
            publishedDate: (0, _dateTimeTypes.dateTimeType)('publishedDate', {
                allowNull: true
            })
        }, {
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            ...options
        });
    }
    static createWithTests(data) {
        return this.sequelize.transaction(async ()=>{
            const { labTestTypeIds = [] } = data;
            if (!labTestTypeIds.length) {
                throw new _errors.InvalidOperationError('A request must have at least one test');
            }
            const { LabTest, LabTestPanelRequest, LabRequestLog } = this.sequelize.models;
            const { labTest, labTestPanelId, userId, ...requestData } = data;
            let newLabRequest;
            if (labTestPanelId) {
                const { id: labTestPanelRequestId } = await LabTestPanelRequest.create({
                    encounterId: data.encounterId,
                    labTestPanelId
                });
                newLabRequest = await this.create({
                    ...requestData,
                    labTestPanelRequestId
                });
            } else {
                newLabRequest = await this.create(requestData);
            }
            await LabRequestLog.create({
                status: newLabRequest.status,
                labRequestId: newLabRequest.id,
                updatedById: userId
            });
            // then create tests
            await Promise.all(labTestTypeIds.map((t)=>LabTest.create({
                    labTestTypeId: t,
                    labRequestId: newLabRequest.id,
                    date: labTest?.date
                })));
            return newLabRequest;
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Department, {
            foreignKey: 'departmentId',
            as: 'department'
        });
        this.belongsTo(models.User, {
            foreignKey: 'collectedById',
            as: 'collectedBy'
        });
        this.belongsTo(models.User, {
            foreignKey: 'requestedById',
            as: 'requestedBy'
        });
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestCategoryId',
            as: 'category'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labSampleSiteId',
            as: 'site'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestPriorityId',
            as: 'priority'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'labTestLaboratoryId',
            as: 'laboratory'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'specimenTypeId',
            as: 'specimenType'
        });
        this.belongsTo(models.LabTestPanelRequest, {
            foreignKey: 'labTestPanelRequestId',
            as: 'labTestPanelRequest'
        });
        this.hasMany(models.LabTest, {
            foreignKey: 'labRequestId',
            as: 'tests'
        });
        this.hasMany(models.CertificateNotification, {
            foreignKey: 'labRequestId',
            as: 'certificate_notification'
        });
        this.hasMany(models.LabRequestAttachment, {
            foreignKey: 'labRequestId',
            as: 'labRequestAttachments'
        });
        this.hasMany(models.Note, {
            foreignKey: 'recordId',
            as: 'notes',
            constraints: false,
            scope: {
                recordType: this.name
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'department',
            'requestedBy',
            'category',
            'priority',
            'laboratory',
            'site',
            'collectedBy',
            'specimenType',
            {
                association: 'labTestPanelRequest',
                include: [
                    'labTestPanel'
                ]
            },
            {
                association: 'tests',
                include: [
                    'labTestType'
                ]
            }
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable, sessionConfig) {
        if (sessionConfig.syncAllLabRequests) {
            return ''; // include all lab requests
        }
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: 'encounters.patient_id',
                encounterId: `${this.tableName}.encounter_id`,
                isLabRequestValue: 'TRUE'
            }),
            joins: (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
                this.tableName,
                'encounters'
            ])
        };
    }
    getTests() {
        return this.sequelize.models.LabTest.findAll({
            where: {
                labRequestId: this.id
            }
        });
    }
    getLatestAttachment() {
        return this.sequelize.models.LabRequestAttachment.findOne({
            where: {
                labRequestId: this.id,
                replacedById: null
            }
        });
    }
};

//# sourceMappingURL=LabRequest.js.map