"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "PatientDeathData", {
    enumerable: true,
    get: function() {
        return PatientDeathData;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _dateTimeTypes = require("./dateTimeTypes");
const _Model = require("./Model");
const _buildPatientSyncFilterViaPatientId = require("./buildPatientSyncFilterViaPatientId");
const _onSaveMarkPatientForSync = require("./onSaveMarkPatientForSync");
const _buildPatientLinkedLookupFilter = require("./buildPatientLinkedLookupFilter");
let PatientDeathData = class PatientDeathData extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            birthWeight: {
                type: _sequelize.Sequelize.INTEGER,
                unsigned: true
            },
            carrierAge: {
                type: _sequelize.Sequelize.INTEGER,
                unsigned: true
            },
            carrierPregnancyWeeks: {
                type: _sequelize.Sequelize.INTEGER,
                unsigned: true
            },
            externalCauseDate: (0, _dateTimeTypes.dateType)('externalCauseDate'),
            lastSurgeryDate: (0, _dateTimeTypes.dateType)('lastSurgeryDate'),
            externalCauseLocation: _sequelize.Sequelize.STRING,
            externalCauseNotes: _sequelize.Sequelize.TEXT,
            fetalOrInfant: _sequelize.Sequelize.BOOLEAN,
            hoursSurvivedSinceBirth: {
                type: _sequelize.Sequelize.INTEGER,
                unsigned: true
            },
            manner: _sequelize.Sequelize.STRING,
            pregnancyContributed: _sequelize.Sequelize.STRING,
            recentSurgery: _sequelize.Sequelize.STRING,
            stillborn: _sequelize.Sequelize.STRING,
            wasPregnant: _sequelize.Sequelize.STRING,
            withinDayOfBirth: _sequelize.Sequelize.BOOLEAN,
            outsideHealthFacility: _sequelize.Sequelize.BOOLEAN,
            primaryCauseTimeAfterOnset: _sequelize.Sequelize.INTEGER,
            antecedentCause1TimeAfterOnset: _sequelize.Sequelize.INTEGER,
            antecedentCause2TimeAfterOnset: _sequelize.Sequelize.INTEGER,
            antecedentCause3TimeAfterOnset: _sequelize.Sequelize.INTEGER,
            isFinal: _sequelize.Sequelize.BOOLEAN,
            visibilityStatus: {
                type: _sequelize.Sequelize.TEXT,
                defaultValue: _constants.VISIBILITY_STATUSES.CURRENT
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            tableName: 'patient_death_data',
            validate: {
                mustHavePatient () {
                    if (this.deletedAt) return;
                    if (!this.patientId) {
                        throw new _errors.InvalidOperationError('Patient death data must have a patient.');
                    }
                },
                mustHaveClinician () {
                    if (this.deletedAt) return;
                    if (!this.clinicianId) {
                        throw new _errors.InvalidOperationError('Patient death data must have a clinician.');
                    }
                },
                yesNoUnknownFields () {
                    if (this.deletedAt) return;
                    for (const field of [
                        'recentSurgery',
                        'wasPregnant',
                        'pregnancyContributed',
                        'stillborn'
                    ]){
                        if (this[field] && ![
                            'yes',
                            'no',
                            'unknown'
                        ].includes(this[field])) {
                            throw new _errors.InvalidOperationError(`${field} must be 'yes', 'no', 'unknown', or null`);
                        }
                    }
                }
            }
        });
        (0, _onSaveMarkPatientForSync.onSaveMarkPatientForSync)(this);
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            foreignKey: 'patientId'
        });
        this.belongsTo(models.User, {
            foreignKey: 'clinicianId',
            as: 'clinician'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        // conceptually "hasOne" but we want the foreign key to be here
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'primaryCauseConditionId',
            as: 'primaryCauseCondition'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause1ConditionId',
            as: 'antecedentCause1Condition',
            allowNull: true
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause2ConditionId',
            as: 'antecedentCause2Condition',
            allowNull: true
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'antecedentCause3ConditionId',
            as: 'antecedentCause3Condition',
            allowNull: true
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'lastSurgeryReasonId',
            as: 'lastSurgeryReason'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'carrierExistingConditionId',
            as: 'carrierExistingCondition'
        });
        this.hasMany(models.ContributingDeathCause, {
            foreignKey: 'patientDeathDataId',
            as: 'contributingCauses'
        });
    }
    static buildSyncLookupQueryDetails() {
        return (0, _buildPatientLinkedLookupFilter.buildPatientLinkedLookupFilter)(this);
    }
    static buildPatientSyncFilter = _buildPatientSyncFilterViaPatientId.buildPatientSyncFilterViaPatientId;
};

//# sourceMappingURL=PatientDeathData.js.map