"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "ReportRequest", {
    enumerable: true,
    get: function() {
        return ReportRequest;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _logging = require("../services/logging");
const _errors = require("../errors");
const _Model = require("./Model");
let ReportRequest = class ReportRequest extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            reportType: {
                type: _sequelize.Sequelize.STRING
            },
            recipients: {
                type: _sequelize.Sequelize.TEXT,
                allowNull: false
            },
            parameters: _sequelize.Sequelize.TEXT,
            status: {
                type: _sequelize.Sequelize.ENUM(_constants.REPORT_REQUEST_STATUS_VALUES),
                allowNull: false
            },
            exportFormat: {
                type: _sequelize.Sequelize.ENUM(Object.values(_constants.REPORT_EXPORT_FORMATS)),
                allowNull: false,
                defaultValue: _constants.REPORT_EXPORT_FORMATS.XLSX
            },
            error: _sequelize.Sequelize.TEXT,
            processStartedTime: _sequelize.Sequelize.DATE
        }, {
            ...options,
            validate: {
                // Must have
                hasReportId: ()=>{
                    // No validation on deleted records
                    if (!this.deletedAt) return;
                    if (!this.reportDefinitionVersionId && !this.reportType) {
                        throw new _errors.InvalidOperationError('A report request must have either a reportType or a reportDefinitionVersionId');
                    }
                    if (this.reportDefinitionVersionId && this.reportType) {
                        throw new _errors.InvalidOperationError('A report request must have either a reportType or a reportDefinitionVersionId, not both');
                    }
                }
            },
            syncDirection: _constants.SYNC_DIRECTIONS.PUSH_TO_CENTRAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: {
                name: 'requestedByUserId',
                allowNull: false
            },
            onDelete: 'CASCADE'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        this.belongsTo(models.ReportDefinitionVersion, {
            foreignKey: 'reportDefinitionVersionId',
            as: 'reportDefinitionVersion'
        });
    }
    getReportId() {
        return this.reportDefinitionVersionId || this.reportType;
    }
    getParameters() {
        try {
            return JSON.parse(this.parameters);
        } catch (e) {
            _logging.log.warn(`Failed to parse ReportRequest parameters ${e}`);
            return {};
        }
    }
    getRecipients() {
        try {
            return JSON.parse(this.recipients);
        } catch (e) {
            // Backwards compatibility: support previous syntax of plain string
            return {
                email: this.recipients.split(',')
            };
        }
    }
};

//# sourceMappingURL=ReportRequest.js.map