"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.GrpcNativeInstrumentation = void 0;
const instrumentation_1 = require("@opentelemetry/instrumentation");
const api_1 = require("@opentelemetry/api");
const serverUtils_1 = require("./serverUtils");
const clientUtils_1 = require("./clientUtils");
const utils_1 = require("../utils");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const AttributeValues_1 = require("../enums/AttributeValues");
/**
 * Holding reference to grpc module here to access constant of grpc modules
 * instead of just requiring it avoid directly depending on grpc itself.
 */
let grpcClient;
class GrpcNativeInstrumentation extends instrumentation_1.InstrumentationBase {
    constructor(name, version, config) {
        super(name, version, config);
        this._metadataCapture = this._createMetadataCapture();
    }
    init() {
        return [
            new instrumentation_1.InstrumentationNodeModuleDefinition('grpc', ['1.*'], (moduleExports, version) => {
                this._diag.debug(`Applying patch for grpc@${version}`);
                grpcClient = moduleExports;
                if ((0, instrumentation_1.isWrapped)(moduleExports.Server.prototype.register)) {
                    this._unwrap(moduleExports.Server.prototype, 'register');
                }
                this._wrap(moduleExports.Server.prototype, 'register', this._patchServer());
                // Wrap the externally exported client constructor
                if ((0, instrumentation_1.isWrapped)(moduleExports.makeGenericClientConstructor)) {
                    this._unwrap(moduleExports, 'makeGenericClientConstructor');
                }
                this._wrap(moduleExports, 'makeGenericClientConstructor', this._patchClient());
                return moduleExports;
            }, (moduleExports, version) => {
                if (moduleExports === undefined)
                    return;
                this._diag.debug(`Removing patch for grpc@${version}`);
                this._unwrap(moduleExports.Server.prototype, 'register');
            }, this._getInternalPatchs()),
        ];
    }
    getConfig() {
        return super.getConfig();
    }
    setConfig(config) {
        super.setConfig(config);
        this._metadataCapture = this._createMetadataCapture();
    }
    _getInternalPatchs() {
        const onPatch = (moduleExports, version) => {
            this._diag.debug(`Applying internal patch for grpc@${version}`);
            if ((0, instrumentation_1.isWrapped)(moduleExports.makeClientConstructor)) {
                this._unwrap(moduleExports, 'makeClientConstructor');
            }
            this._wrap(moduleExports, 'makeClientConstructor', this._patchClient());
            return moduleExports;
        };
        const onUnPatch = (moduleExports, version) => {
            if (moduleExports === undefined)
                return;
            this._diag.debug(`Removing internal patch for grpc@${version}`);
            this._unwrap(moduleExports, 'makeClientConstructor');
        };
        return [
            new instrumentation_1.InstrumentationNodeModuleFile('grpc/src/node/src/client.js', ['0.13 - 1.6'], onPatch, onUnPatch),
            new instrumentation_1.InstrumentationNodeModuleFile('grpc/src/client.js', ['^1.7'], onPatch, onUnPatch),
        ];
    }
    _patchServer() {
        const instrumentation = this;
        return (originalRegister) => {
            instrumentation._diag.debug('patched gRPC server');
            return function register(name, handler, serialize, deserialize, type) {
                const originalResult = originalRegister.apply(this, arguments);
                const handlerSet = this.handlers[name];
                instrumentation._wrap(handlerSet, 'func', (originalFunc) => {
                    return function func(call, callback) {
                        const self = this;
                        if (serverUtils_1.shouldNotTraceServerCall.call(instrumentation, call, name)) {
                            switch (type) {
                                case 'unary':
                                case 'client_stream':
                                    return originalFunc.call(self, call, callback);
                                case 'server_stream':
                                case 'bidi':
                                    return originalFunc.call(self, call);
                                default:
                                    return originalResult;
                            }
                        }
                        const spanName = `grpc.${name.replace('/', '')}`;
                        const spanOptions = {
                            kind: api_1.SpanKind.SERVER,
                        };
                        instrumentation._diag.debug(`patch func: ${JSON.stringify(spanOptions)}`);
                        api_1.context.with(api_1.propagation.extract(api_1.context.active(), call.metadata, {
                            get: (metadata, key) => metadata.get(key).map(String),
                            keys: metadata => Object.keys(metadata.getMap()),
                        }), () => {
                            const { service, method } = (0, utils_1._extractMethodAndService)(name);
                            const span = instrumentation.tracer
                                .startSpan(spanName, spanOptions)
                                .setAttributes({
                                [semantic_conventions_1.SemanticAttributes.RPC_SYSTEM]: AttributeValues_1.AttributeValues.RPC_SYSTEM,
                                [semantic_conventions_1.SemanticAttributes.RPC_METHOD]: method,
                                [semantic_conventions_1.SemanticAttributes.RPC_SERVICE]: service,
                            });
                            instrumentation._metadataCapture.server.captureRequestMetadata(span, call.metadata);
                            instrumentation._wrap(call, 'sendMetadata', originalSendMetadata => (responseMetadata) => {
                                instrumentation._metadataCapture.server.captureResponseMetadata(span, responseMetadata);
                                originalSendMetadata.call(call, responseMetadata);
                            });
                            api_1.context.with(api_1.trace.setSpan(api_1.context.active(), span), () => {
                                switch (type) {
                                    case 'unary':
                                    case 'client_stream':
                                        return (0, serverUtils_1.clientStreamAndUnaryHandler)(span, call, callback, originalFunc, self);
                                    case 'server_stream':
                                    case 'bidi':
                                        return (0, serverUtils_1.serverStreamAndBidiHandler)(span, call, originalFunc, self);
                                    default:
                                        break;
                                }
                            });
                        });
                    };
                });
                return originalResult;
            };
        };
    }
    _patchClient() {
        const instrumentation = this;
        return (original) => {
            instrumentation._diag.debug('patching client');
            return function makeClientConstructor(methods, _serviceName, _options) {
                const client = original.apply(this, arguments);
                instrumentation._massWrap(client.prototype, instrumentation._getMethodsToWrap(client, methods), instrumentation._getPatchedClientMethods());
                return client;
            };
        };
    }
    _getMethodsToWrap(client, methods) {
        const methodList = [];
        // For a method defined in .proto as "UnaryMethod"
        Object.entries(methods).forEach(([name, { originalName }]) => {
            if (!(0, utils_1._methodIsIgnored)(name, this.getConfig().ignoreGrpcMethods)) {
                methodList.push(name); // adds camel case method name: "unaryMethod"
                if (originalName &&
                    // eslint-disable-next-line no-prototype-builtins
                    client.prototype.hasOwnProperty(originalName) &&
                    name !== originalName // do not add duplicates
                ) {
                    // adds original method name: "UnaryMethod",
                    methodList.push(originalName);
                }
            }
        });
        return methodList;
    }
    _getPatchedClientMethods() {
        const instrumentation = this;
        return (original) => {
            instrumentation._diag.debug('patch all client methods');
            function clientMethodTrace() {
                var _a;
                const name = `grpc.${(_a = original.path) === null || _a === void 0 ? void 0 : _a.replace('/', '')}`;
                const args = Array.prototype.slice.call(arguments);
                const metadata = (0, clientUtils_1.getMetadata)(grpcClient, original, args);
                const { service, method } = (0, utils_1._extractMethodAndService)(original.path);
                const span = instrumentation.tracer
                    .startSpan(name, {
                    kind: api_1.SpanKind.CLIENT,
                })
                    .setAttributes({
                    [semantic_conventions_1.SemanticAttributes.RPC_SYSTEM]: AttributeValues_1.AttributeValues.RPC_SYSTEM,
                    [semantic_conventions_1.SemanticAttributes.RPC_METHOD]: method,
                    [semantic_conventions_1.SemanticAttributes.RPC_SERVICE]: service,
                });
                // set net.peer.* from target (e.g., "dns:otel-productcatalogservice:8080") as a hint to APMs
                const parsedUri = utils_1.URI_REGEX.exec(this.getChannel().getTarget());
                if (parsedUri != null && parsedUri.groups != null) {
                    span.setAttribute(semantic_conventions_1.SemanticAttributes.NET_PEER_NAME, parsedUri.groups['name']);
                    span.setAttribute(semantic_conventions_1.SemanticAttributes.NET_PEER_PORT, parseInt(parsedUri.groups['port']));
                }
                instrumentation._metadataCapture.client.captureRequestMetadata(span, metadata);
                return api_1.context.with(api_1.trace.setSpan(api_1.context.active(), span), () => (0, clientUtils_1.makeGrpcClientRemoteCall)(instrumentation._metadataCapture, original, args, metadata, this)(span));
            }
            Object.assign(clientMethodTrace, original);
            return clientMethodTrace;
        };
    }
    _createMetadataCapture() {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        const config = this.getConfig();
        return {
            client: {
                captureRequestMetadata: (0, utils_1.metadataCapture)('request', (_c = (_b = (_a = config.metadataToSpanAttributes) === null || _a === void 0 ? void 0 : _a.client) === null || _b === void 0 ? void 0 : _b.requestMetadata) !== null && _c !== void 0 ? _c : []),
                captureResponseMetadata: (0, utils_1.metadataCapture)('response', (_f = (_e = (_d = config.metadataToSpanAttributes) === null || _d === void 0 ? void 0 : _d.client) === null || _e === void 0 ? void 0 : _e.responseMetadata) !== null && _f !== void 0 ? _f : []),
            },
            server: {
                captureRequestMetadata: (0, utils_1.metadataCapture)('request', (_j = (_h = (_g = config.metadataToSpanAttributes) === null || _g === void 0 ? void 0 : _g.server) === null || _h === void 0 ? void 0 : _h.requestMetadata) !== null && _j !== void 0 ? _j : []),
                captureResponseMetadata: (0, utils_1.metadataCapture)('response', (_m = (_l = (_k = config.metadataToSpanAttributes) === null || _k === void 0 ? void 0 : _k.server) === null || _l === void 0 ? void 0 : _l.responseMetadata) !== null && _m !== void 0 ? _m : []),
            },
        };
    }
}
exports.GrpcNativeInstrumentation = GrpcNativeInstrumentation;
//# sourceMappingURL=index.js.map