"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "encounter", {
    enumerable: true,
    get: function() {
        return encounter;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _errors = require("@tamanu/shared/errors");
const _dateTime = require("@tamanu/shared/utils/dateTime");
const _constants = require("@tamanu/constants");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _uploadAttachment = require("../../utils/uploadAttachment");
const _routeHandlers = require("../../routeHandlers");
const _createPatientLetter = require("../../routeHandlers/createPatientLetter");
const _labs = require("../../routeHandlers/labs");
const _deleteModel = require("../../routeHandlers/deleteModel");
const _getPermittedSurveyIds = require("../../utils/getPermittedSurveyIds");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const encounter = (0, _crudHelpers.softDeletionCheckingRouter)('Encounter');
encounter.get('/:id', (0, _crudHelpers.simpleGet)('Encounter'));
encounter.post('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body, user } = req;
    req.checkPermission('create', 'Encounter');
    const encounterObject = await models.Encounter.create({
        ...body,
        actorId: user.id
    });
    if (body.dietIds) {
        const dietIds = JSON.parse(body.dietIds);
        await encounterObject.addDiets(dietIds);
    }
    res.send(encounterObject);
}));
encounter.put('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, user, params } = req;
    const { referralId, id } = params;
    req.checkPermission('read', 'Encounter');
    const encounterObject = await models.Encounter.findByPk(id);
    if (!encounterObject) throw new _errors.NotFoundError();
    req.checkPermission('write', encounterObject);
    await db.transaction(async ()=>{
        let systemNote;
        if (req.body.discharge) {
            req.checkPermission('write', 'Discharge');
            if (!req.body.discharge.dischargerId) {
                // Only automatic discharges can have a null discharger ID
                throw new _errors.InvalidParameterError('A discharge must have a discharger.');
            }
            const discharger = await models.User.findByPk(req.body.discharge.dischargerId);
            if (!discharger) {
                throw new _errors.InvalidParameterError(`Discharger with id ${req.body.discharge.dischargerId} not found.`);
            }
            systemNote = `Patient discharged by ${discharger.displayName}.`;
            // Update medications that were marked for discharge and ensure
            // only isDischarge, quantity and repeats fields are edited
            const medications = req.body.medications || {};
            for (const [medicationId, medicationValues] of Object.entries(medications)){
                const { isDischarge, quantity, repeats } = medicationValues;
                if (isDischarge) {
                    const medication = await models.EncounterMedication.findByPk(medicationId);
                    await medication.update({
                        isDischarge,
                        quantity,
                        repeats
                    });
                }
            }
        }
        if (referralId) {
            const referral = await models.Referral.findByPk(referralId, {
                paranoid: false
            });
            if (referral && referral.deletedAt) throw new _errors.InvalidOperationError('Cannot update a deleted referral.');
            await referral.update({
                encounterId: id
            });
        }
        if (req.body.dietIds) {
            const dietIds = JSON.parse(req.body.dietIds);
            await encounterObject.setDiets(dietIds);
        }
        await encounterObject.update({
            ...req.body,
            systemNote
        }, user);
    });
    res.send(encounterObject);
}));
encounter.post('/:id/notes', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body, params } = req;
    const { id } = params;
    req.checkPermission('write', 'Encounter');
    const owner = await models.Encounter.findByPk(id);
    if (!owner) {
        throw new _errors.NotFoundError();
    }
    req.checkPermission('write', owner);
    const note = await owner.createNote(body);
    res.send(note);
}));
encounter.post('/:id/documentMetadata', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    // TODO: figure out permissions with Attachment and DocumentMetadata
    req.checkPermission('write', 'DocumentMetadata');
    // Make sure the specified encounter exists
    const specifiedEncounter = await models.Encounter.findByPk(params.id);
    if (!specifiedEncounter) {
        throw new _errors.NotFoundError();
    }
    // Create file on the central server
    const { attachmentId, type, metadata } = await (0, _uploadAttachment.uploadAttachment)(req, _constants.DOCUMENT_SIZE_LIMIT);
    const documentMetadataObject = await models.DocumentMetadata.create({
        ...metadata,
        attachmentId,
        type,
        encounterId: params.id,
        documentUploadedAt: (0, _dateTime.getCurrentDateTimeString)(),
        source: _constants.DOCUMENT_SOURCES.UPLOADED
    });
    res.send(documentMetadataObject);
}));
encounter.post('/:id/createPatientLetter', (0, _createPatientLetter.createPatientLetter)('Encounter', 'encounterId'));
encounter.delete('/:id/documentMetadata/:documentMetadataId', _deleteModel.deleteDocumentMetadata);
encounter.delete('/:id', _deleteModel.deleteEncounter);
const encounterRelations = (0, _crudHelpers.permissionCheckingRouter)('read', 'Encounter');
encounterRelations.get('/:id/discharge', (0, _crudHelpers.simpleGetHasOne)('Discharge', 'encounterId'));
encounterRelations.get('/:id/legacyVitals', (0, _crudHelpers.simpleGetList)('Vitals', 'encounterId'));
encounterRelations.get('/:id/diagnoses', (0, _crudHelpers.simpleGetList)('EncounterDiagnosis', 'encounterId'));
encounterRelations.get('/:id/medications', (0, _crudHelpers.simpleGetList)('EncounterMedication', 'encounterId'));
encounterRelations.get('/:id/procedures', (0, _crudHelpers.simpleGetList)('Procedure', 'encounterId'));
encounterRelations.get('/:id/labRequests', (0, _labs.getLabRequestList)('encounterId', {
    additionalFilters: {
        status: {
            [_sequelize.Op.notIn]: [
                _constants.LAB_REQUEST_STATUSES.DELETED,
                _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR
            ]
        }
    }
}));
encounterRelations.get('/:id/referral', (0, _crudHelpers.simpleGetList)('Referral', 'encounterId'));
encounterRelations.get('/:id/triages', (0, _crudHelpers.simpleGetList)('Triage', 'encounterId'));
encounterRelations.get('/:id/documentMetadata', (0, _crudHelpers.paginatedGetList)('DocumentMetadata', 'encounterId'));
encounterRelations.get('/:id/imagingRequests', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query } = req;
    const { ImagingRequest } = models;
    const { id: encounterId } = params;
    const { order = 'ASC', orderBy = 'createdAt', rowsPerPage, page, includeNotes: includeNotesStr = 'true', status } = query;
    const includeNote = includeNotesStr === 'true';
    req.checkPermission('list', 'ImagingRequest');
    const associations = ImagingRequest.getListReferenceAssociations() || [];
    const baseQueryOptions = {
        where: {
            encounterId,
            status: status || {
                [_sequelize.Op.notIn]: [
                    _constants.IMAGING_REQUEST_STATUS_TYPES.DELETED,
                    _constants.IMAGING_REQUEST_STATUS_TYPES.ENTERED_IN_ERROR
                ]
            }
        },
        order: orderBy ? [
            [
                ...orderBy.split('.'),
                order.toUpperCase()
            ]
        ] : undefined,
        include: associations
    };
    const count = await ImagingRequest.count({
        ...baseQueryOptions
    });
    const objects = await ImagingRequest.findAll({
        ...baseQueryOptions,
        limit: rowsPerPage,
        offset: page && rowsPerPage ? page * rowsPerPage : undefined
    });
    const data = await Promise.all(objects.map(async (ir)=>{
        return {
            ...ir.forResponse(),
            ...includeNote ? await ir.extractNotes() : undefined,
            areas: ir.areas.map((a)=>a.forResponse()),
            results: ir.results.map((result)=>result.forResponse())
        };
    }));
    res.send({
        count,
        data
    });
}));
encounterRelations.get('/:id/notes', (0, _routeHandlers.noteListHandler)(_constants.NOTE_RECORD_TYPES.ENCOUNTER));
encounterRelations.get('/:id/notes/noteTypes', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const encounterId = params.id;
    const noteTypeCounts = await models.Note.count({
        group: [
            'noteType'
        ],
        where: {
            recordId: encounterId,
            recordType: 'Encounter'
        }
    });
    const noteTypeToCount = {};
    noteTypeCounts.forEach((n)=>{
        noteTypeToCount[n.noteType] = n.count;
    });
    res.send({
        data: noteTypeToCount
    });
}));
encounterRelations.get('/:id/notes/:noteId/changelogs', (0, _routeHandlers.noteChangelogsHandler)(_constants.NOTE_RECORD_TYPES.ENCOUNTER));
encounterRelations.get('/:id/invoice', (0, _crudHelpers.simpleGetHasOne)('Invoice', 'encounterId', {}));
const PROGRAM_RESPONSE_SORT_KEYS = {
    endTime: 'end_time',
    submittedBy: 'submitted_by',
    programName: 'program_name',
    surveyName: 'survey_name',
    resultText: 'result_text'
};
encounterRelations.get('/:id/programResponses', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, params, query } = req;
    req.checkPermission('list', 'SurveyResponse');
    const encounterId = params.id;
    const surveyType = 'programs';
    const { order = 'asc', orderBy = 'endTime' } = query;
    const sortKey = PROGRAM_RESPONSE_SORT_KEYS[orderBy] || PROGRAM_RESPONSE_SORT_KEYS.endTime;
    const sortDirection = order.toLowerCase() === 'asc' ? 'ASC' : 'DESC';
    const permittedSurveyIds = await (0, _getPermittedSurveyIds.getPermittedSurveyIds)(req, models);
    if (!permittedSurveyIds.length) {
        res.send({
            data: [],
            count: 0
        });
    }
    const { count, data } = await (0, _crudHelpers.runPaginatedQuery)(db, models.SurveyResponse, `
        SELECT COUNT(1) as count
        FROM
          survey_responses
          LEFT JOIN encounters
            ON (survey_responses.encounter_id = encounters.id)
          LEFT JOIN surveys
            ON (survey_responses.survey_id = surveys.id)
        WHERE
          survey_responses.encounter_id = :encounterId
        AND
          surveys.survey_type = :surveyType
        AND 
          surveys.id IN (:surveyIds)
        AND
          encounters.deleted_at IS NULL
        AND
          survey_responses.deleted_at IS NULL
      `, `
        SELECT
          survey_responses.*,
          surveys.name as survey_name,
          programs.name as program_name,
          COALESCE(survey_user.display_name, encounter_user.display_name) as submitted_by
        FROM
          survey_responses
          LEFT JOIN surveys
            ON (survey_responses.survey_id = surveys.id)
          LEFT JOIN programs
            ON (programs.id = surveys.program_id)
          LEFT JOIN encounters
            ON (encounters.id = survey_responses.encounter_id)
          LEFT JOIN users encounter_user
            ON (encounter_user.id = encounters.examiner_id)
          LEFT JOIN users survey_user
            ON (survey_user.id = survey_responses.user_id)
        WHERE
          survey_responses.encounter_id = :encounterId
        AND
          surveys.survey_type = :surveyType
        AND 
          surveys.id IN (:surveyIds)
        AND
          survey_responses.deleted_at IS NULL
        AND
          encounters.deleted_at is null
        ORDER BY ${sortKey} ${sortDirection}
      `, {
        encounterId,
        surveyType,
        surveyIds: permittedSurveyIds
    }, query);
    res.send({
        count: parseInt(count, 10),
        data
    });
}));
encounterRelations.delete('/:id/programResponses/:surveyResponseId', _deleteModel.deleteSurveyResponse);
encounterRelations.get('/:id/vitals', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, params, query } = req;
    req.checkPermission('list', 'Vitals');
    const encounterId = params.id;
    const { order = 'DESC' } = query;
    // The LIMIT and OFFSET occur in an unusual place in this query
    // So we can't run it through the generic runPaginatedQuery function
    const countResult = await db.query(`
        SELECT COUNT(1) AS count
        FROM survey_response_answers
        INNER JOIN survey_responses response
        ON response.id = response_id
        WHERE data_element_id = :dateDataElement
        AND body IS NOT NULL
        AND response.encounter_id = :encounterId
        AND response.deleted_at IS NULL
      `, {
        replacements: {
            encounterId,
            dateDataElement: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded
        },
        type: _sequelize.QueryTypes.SELECT
    });
    const { count } = countResult[0];
    if (count === 0) {
        res.send({
            data: [],
            count: 0
        });
        return;
    }
    const { page = 0, rowsPerPage = 10 } = query;
    const result = await db.query(`
        WITH
        date AS (
          SELECT response_id, body
          FROM survey_response_answers
          INNER JOIN survey_responses response
          ON response.id = response_id
          WHERE data_element_id = :dateDataElement
          AND body IS NOT NULL
          AND response.encounter_id = :encounterId
          AND response.deleted_at IS NULL
          ORDER BY body ${order} LIMIT :limit OFFSET :offset
        ),
        history AS (
          SELECT
            vl.answer_id,
            ARRAY_AGG((
              JSONB_BUILD_OBJECT(
                'newValue', vl.new_value,
                'reasonForChange', vl.reason_for_change,
                'date', vl.date,
                'userDisplayName', u.display_name
              )
            )) logs
          FROM survey_response_answers sra
          	INNER JOIN survey_responses sr ON sr.id = sra.response_id
          	LEFT JOIN vital_logs vl ON vl.answer_id = sra.id
          	LEFT JOIN users u ON u.id = vl.recorded_by_id
          WHERE sr.encounter_id = :encounterId
          	AND sr.deleted_at IS NULL
          GROUP BY vl.answer_id
        )

        SELECT
          JSONB_BUILD_OBJECT(
            'dataElementId', answer.data_element_id,
            'records', JSONB_OBJECT_AGG(date.body, JSONB_BUILD_OBJECT('id', answer.id, 'body', answer.body, 'logs', history.logs))
          ) result
        FROM
          survey_response_answers answer
        INNER JOIN
          date
        ON date.response_id = answer.response_id
        LEFT JOIN
          history
        ON history.answer_id = answer.id
        GROUP BY answer.data_element_id
      `, {
        replacements: {
            encounterId,
            limit: rowsPerPage,
            offset: page * rowsPerPage,
            dateDataElement: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded
        },
        type: _sequelize.QueryTypes.SELECT
    });
    const data = result.map((r)=>r.result);
    res.send({
        count: parseInt(count, 10),
        data
    });
}));
encounterRelations.get('/:id/vitals/:dataElementId', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query } = req;
    req.checkPermission('list', 'Vitals');
    const { id: encounterId, dataElementId } = params;
    const { startDate, endDate } = query;
    const { SurveyResponse, SurveyResponseAnswer } = models;
    const dateAnswers = await SurveyResponseAnswer.findAll({
        include: [
            {
                model: SurveyResponse,
                required: true,
                as: 'surveyResponse',
                where: {
                    encounterId
                }
            }
        ],
        where: {
            dataElementId: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded,
            body: {
                [_sequelize.Op.gte]: startDate,
                [_sequelize.Op.lte]: endDate
            }
        }
    });
    const responseIds = dateAnswers.map((dateAnswer)=>dateAnswer.responseId);
    const answers = await SurveyResponseAnswer.findAll({
        where: {
            responseId: responseIds,
            dataElementId,
            body: {
                [_sequelize.Op.and]: [
                    {
                        [_sequelize.Op.ne]: ''
                    },
                    {
                        [_sequelize.Op.not]: null
                    }
                ]
            }
        }
    });
    const data = answers.map((answer)=>{
        const { responseId } = answer;
        const recordedDateAnswer = dateAnswers.find((dateAnswer)=>dateAnswer.responseId === responseId);
        const recordedDate = recordedDateAnswer.body;
        return {
            ...answer.dataValues,
            recordedDate
        };
    }).sort((a, b)=>{
        return a.recordedDate > b.recordedDate ? 1 : -1;
    });
    res.send({
        count: data.length,
        data
    });
}));
encounter.use(encounterRelations);

//# sourceMappingURL=encounter.js.map