"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "User", {
    enumerable: true,
    get: function() {
        return User;
    }
});
const _bcrypt = require("bcrypt");
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const DEFAULT_SALT_ROUNDS = 10;
let User = class User extends _Model.Model {
    static SALT_ROUNDS = DEFAULT_SALT_ROUNDS;
    static hashPassword(pw) {
        return (0, _bcrypt.hash)(pw, User.SALT_ROUNDS ?? DEFAULT_SALT_ROUNDS);
    }
    static getSystemUser() {
        return this.findByPk(_constants.SYSTEM_USER_UUID);
    }
    forResponse() {
        const values = Object.assign({}, this.dataValues);
        delete values.password;
        return values;
    }
    async setPassword(pw) {
        this.password = await User.hashPassword(pw);
    }
    static async sanitizeForInsert(values) {
        const { password, ...otherValues } = values;
        if (!password) return values;
        return {
            ...otherValues,
            password: await User.hashPassword(password)
        };
    }
    static async update(values, ...args) {
        const sanitizedValues = await this.sanitizeForInsert(values);
        return super.update(sanitizedValues, ...args);
    }
    static async create(values, ...args) {
        const sanitizedValues = await this.sanitizeForInsert(values);
        return super.create(sanitizedValues, ...args);
    }
    static async bulkCreate(records, ...args) {
        const sanitizedRecords = await Promise.all(records.map((r)=>this.sanitizeForInsert(r)));
        return super.bulkCreate(sanitizedRecords, ...args);
    }
    static async upsert(values, ...args) {
        const sanitizedValues = await this.sanitizeForInsert(values);
        return super.upsert(sanitizedValues, ...args);
    }
    static async getForAuthByEmail(email) {
        // gets the user, as a plain object, with password hash, for use in auth
        const user = await this.scope('withPassword').findOne({
            where: {
                // email addresses are case insensitive so compare them as such
                email: _sequelize.Sequelize.where(_sequelize.Sequelize.fn('lower', _sequelize.Sequelize.col('email')), _sequelize.Sequelize.fn('lower', email)),
                visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
            }
        });
        if (!user) {
            return null;
        }
        return user.get({
            plain: true
        });
    }
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            displayId: _sequelize.Sequelize.STRING,
            email: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false,
                unique: true
            },
            password: _sequelize.Sequelize.STRING,
            displayName: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false
            },
            role: {
                type: _sequelize.Sequelize.STRING,
                defaultValue: 'practitioner',
                allowNull: false
            },
            phoneNumber: {
                type: _sequelize.Sequelize.STRING
            },
            visibilityStatus: {
                type: _sequelize.Sequelize.STRING,
                defaultValue: _constants.VISIBILITY_STATUSES.CURRENT
            }
        }, {
            ...options,
            defaultScope: {
                attributes: {
                    exclude: [
                        'password'
                    ]
                }
            },
            scopes: {
                withPassword: {
                    attributes: {
                        include: [
                            'password'
                        ]
                    }
                }
            },
            indexes: [
                {
                    fields: [
                        'email'
                    ]
                }
            ],
            syncDirection: _constants.SYNC_DIRECTIONS.PULL_FROM_CENTRAL,
            hooks: {
                async beforeUpdate (user) {
                    if (user.changed('password')) {
                        // eslint-disable-next-line require-atomic-updates
                        user.password = await User.hashPassword(user.password);
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.hasMany(models.Discharge, {
            foreignKey: 'dischargerId',
            as: 'discharges'
        });
        this.hasMany(models.ImagingRequest, {
            foreignKey: 'completedById'
        });
        this.hasMany(models.PatientProgramRegistration, {
            foreignKey: 'clinicianId'
        });
        this.hasMany(models.PatientProgramRegistrationCondition, {
            foreignKey: 'clinicianId'
        });
        this.hasMany(models.PatientProgramRegistrationCondition, {
            foreignKey: 'deletionClinicianId'
        });
        this.hasMany(models.UserPreference, {
            foreignKey: 'userId'
        });
        this.hasMany(models.UserFacility, {
            foreignKey: 'facilityId'
        });
        this.belongsToMany(models.Facility, {
            through: 'UserFacility'
        });
    }
    static buildSyncFilter() {
        return null; // syncs everywhere
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
};

//# sourceMappingURL=User.js.map