"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const _utilities = require("../utilities");
const _dateTime = require("../../utils/dateTime");
const parametersToSqlWhere = (parameters)=>{
    const defaultWhereClause = {
        '$labRequest.lab_test_category_id$': 'labTestCategory-COVID',
        date: {
            [_sequelize.Op.gte]: (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)((0, _datefns.subDays)(new Date(), 30)))
        }
    };
    if (!parameters || !Object.keys(parameters).length) {
        return defaultWhereClause;
    }
    const newParameters = {
        ...parameters
    };
    if (parameters.fromDate) {
        (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)((0, _datefns.parseISO)(parameters.fromDate)));
    }
    if (parameters.toDate) {
        (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    const whereClause = Object.entries(newParameters).filter(([, val])=>val).reduce((where, [key, value])=>{
        const newWhere = {
            ...where
        };
        switch(key){
            case 'village':
                newWhere['$labRequest->encounter->patient.village_id$'] = value;
                break;
            case 'labTestLaboratory':
                newWhere['$labRequest.lab_test_laboratory_id$'] = value;
                break;
            case 'fromDate':
                if (!newWhere.date) {
                    newWhere.date = {};
                }
                newWhere.date[_sequelize.Op.gte] = value;
                break;
            case 'toDate':
                if (!newWhere.date) {
                    newWhere.date = {};
                }
                newWhere.date[_sequelize.Op.lte] = value;
                break;
            default:
                break;
        }
        return newWhere;
    }, defaultWhereClause);
    return whereClause;
};
const dataGenerator = async ({ models }, parameters = {})=>{
    const reportColumnTemplate = [
        {
            title: 'Date',
            accessor: (data)=>(0, _dateTime.format)(data.testDate, 'yyyy/MM/dd')
        },
        {
            title: 'Total tests',
            accessor: (data)=>data.totalRecordCount
        },
        {
            title: 'Negative',
            accessor: (data)=>data.negativeRecordCount
        },
        {
            title: 'Positive',
            accessor: (data)=>data.positiveRecordCount
        },
        {
            title: 'Inconclusive',
            accessor: (data)=>data.inconclusiveRecordCount
        },
        {
            title: 'No result yet',
            accessor: (data)=>data.noResultRecordCount
        }
    ];
    const whereClause = parametersToSqlWhere(parameters);
    const labTestData = await models.LabTest.findAll({
        includeIgnoreAttributes: false,
        attributes: [
            [
                _sequelize.Sequelize.literal(`DATE("date")`),
                'testDate'
            ],
            'result',
            [
                _sequelize.Sequelize.literal(`COUNT(*)`),
                'count'
            ]
        ],
        include: [
            {
                model: models.LabRequest,
                as: 'labRequest',
                attributes: [],
                where: {
                    status: {
                        [_sequelize.Op.notIn]: [
                            _constants.LAB_REQUEST_STATUSES.DELETED,
                            _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR,
                            _constants.LAB_REQUEST_STATUSES.CANCELLED,
                            _constants.LAB_REQUEST_STATUSES.INVALIDATED
                        ]
                    }
                },
                include: [
                    {
                        model: models.Encounter,
                        as: 'encounter',
                        attributes: [],
                        include: [
                            {
                                model: models.Patient,
                                as: 'patient',
                                attributes: [],
                                include: [
                                    {
                                        model: models.ReferenceData,
                                        as: 'village'
                                    }
                                ]
                            }
                        ]
                    },
                    {
                        model: models.ReferenceData,
                        as: 'laboratory'
                    }
                ]
            }
        ],
        where: whereClause,
        group: [
            'testDate',
            'result'
        ],
        order: [
            [
                _sequelize.Sequelize.literal(`"testDate"`),
                'ASC'
            ]
        ]
    });
    const labTestDataByDate = (0, _lodash.groupBy)(labTestData.map((p)=>p.dataValues), 'testDate');
    const reportData = Object.entries(labTestDataByDate).map(([testDate, records])=>{
        const positiveRecord = records.find((r)=>r.result === 'Positive');
        const negativeRecord = records.find((r)=>r.result === 'Negative');
        const inconclusiveRecord = records.find((r)=>r.result === 'Inconclusive');
        const noResultRecord = records.find((r)=>r.result === null || r.result === '');
        const positiveRecordCount = positiveRecord ? Number(positiveRecord.count) : 0;
        const negativeRecordCount = negativeRecord ? Number(negativeRecord.count) : 0;
        const inconclusiveRecordCount = inconclusiveRecord ? Number(inconclusiveRecord.count) : 0;
        const noResultRecordCount = noResultRecord ? Number(noResultRecord.count) : 0;
        const totalRecordCount = positiveRecordCount + negativeRecordCount + inconclusiveRecordCount + noResultRecordCount;
        return {
            testDate,
            positiveRecordCount,
            negativeRecordCount,
            inconclusiveRecordCount,
            noResultRecordCount,
            totalRecordCount
        };
    });
    return (0, _utilities.generateReportFromQueryData)(reportData, reportColumnTemplate);
};

//# sourceMappingURL=covid-swab-lab-tests-summary.js.map