import React from 'react';
import { Document, Image, StyleSheet, View } from '@react-pdf/renderer';
import { getDOB, getSex } from '../patientAccessors';
import JsBarcode from 'jsbarcode';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
const CustomBarcode = ({ id, width, height })=>{
    // eslint-disable-next-line no-undef
    const canvas = document.createElement('canvas');
    JsBarcode(canvas, id, {
        width: 1,
        margin: 0,
        displayValue: false
    });
    const barcode = canvas.toDataURL();
    return /*#__PURE__*/ React.createElement(Image, {
        source: barcode,
        style: {
            height,
            maxWidth: width,
            objectFit: 'cover'
        }
    });
};
const convertToPt = (mm)=>{
    // remove 'mm' etc from strings
    if (typeof mm === 'string') return parseFloat(mm.replace(/[^0-9.]/i, '')) * 2.835;
    return mm * 2.835;
};
const styles = StyleSheet.create({
    mainContainer: {
        display: 'flex',
        flexDirection: 'row',
        height: '28.6mm',
        overflow: 'hidden'
    },
    photoContainer: {
        width: '1in',
        paddingHorizontal: '2mm',
        paddingTop: '1mm'
    },
    photoFrame: {
        width: '1in',
        height: '1.3in'
    },
    photo: {
        width: '100%',
        height: '100%',
        objectFit: 'cover'
    },
    details: {
        display: 'flex',
        flexDirection: 'column',
        justifyContent: 'center',
        marginLeft: '5mm'
    },
    infoRow: {
        display: 'flex',
        flexDirection: 'row',
        fontSize: '2.4mm',
        marginVertical: 1.8
    },
    detailsKey: {
        width: '18mm',
        fontFamily: 'Helvetica-Bold'
    },
    detailsValue: {
        maxWidth: '35mm',
        overflow: 'hidden',
        fontFamily: 'Helvetica-Bold'
    },
    barcodeRow: {
        height: '6.3mm',
        marginLeft: '29.4mm'
    }
});
const MainContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.mainContainer,
        ...props
    });
const PhotoContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.photoContainer,
        ...props
    });
const PhotoFrame = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.photoFrame,
        ...props
    });
const Details = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.details,
        ...props
    });
const InfoRow = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.infoRow,
        ...props
    });
const DetailsKey = (props)=>/*#__PURE__*/ React.createElement(Text, {
        style: styles.detailsKey,
        ...props
    });
const DetailsValue = (props)=>/*#__PURE__*/ React.createElement(Text, {
        style: styles.detailsValue,
        ...props
    });
const BarcodeRow = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: styles.barcodeRow,
        ...props
    });
const DetailsRow = ({ name, value, getLocalisation })=>{
    const label = getLocalisation(`fields.${name}.shortLabel`);
    return /*#__PURE__*/ React.createElement(InfoRow, null, /*#__PURE__*/ React.createElement(DetailsKey, null, `${label}: `), /*#__PURE__*/ React.createElement(DetailsValue, null, value));
};
const PatientPhoto = ({ patientImageData })=>{
    return /*#__PURE__*/ React.createElement(PhotoContainer, null, /*#__PURE__*/ React.createElement(PhotoFrame, null, /*#__PURE__*/ React.createElement(Image, {
        src: `data:image/jpeg;base64,${patientImageData}`,
        style: styles.photo
    })));
};
const IDCardPrintoutComponent = ({ patient, patientImageData, cardDimensions, measures, getLocalisation })=>{
    const pageStyles = StyleSheet.create({
        card: {
            width: cardDimensions.width,
            height: cardDimensions.height,
            marginTop: convertToPt(measures.cardMarginTop),
            marginLeft: convertToPt(measures.cardMarginLeft),
            display: 'flex',
            flexDirection: 'column'
        }
    });
    const Card = (props)=>/*#__PURE__*/ React.createElement(View, {
            style: pageStyles.card,
            ...props
        });
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: {
            paddingTop: convertToPt('10.6mm')
        }
    }, /*#__PURE__*/ React.createElement(Card, null, /*#__PURE__*/ React.createElement(MainContainer, null, /*#__PURE__*/ React.createElement(PatientPhoto, {
        patientImageData: patientImageData
    }), /*#__PURE__*/ React.createElement(Details, null, /*#__PURE__*/ React.createElement(DetailsRow, {
        name: "displayId",
        value: patient.displayId,
        getLocalisation: getLocalisation
    }), /*#__PURE__*/ React.createElement(DetailsRow, {
        name: "lastName",
        value: patient.lastName,
        getLocalisation: getLocalisation
    }), /*#__PURE__*/ React.createElement(DetailsRow, {
        name: "firstName",
        value: patient.firstName,
        getLocalisation: getLocalisation
    }), /*#__PURE__*/ React.createElement(DetailsRow, {
        name: "dateOfBirth",
        value: getDOB(patient),
        getLocalisation: getLocalisation
    }), /*#__PURE__*/ React.createElement(DetailsRow, {
        name: "sex",
        value: getSex(patient),
        getLocalisation: getLocalisation
    }))), /*#__PURE__*/ React.createElement(BarcodeRow, null, /*#__PURE__*/ React.createElement(CustomBarcode, {
        height: "5.9mm",
        width: "33mm",
        id: patient.displayId
    })))));
};
export const IDCardPrintout = withLanguageContext(IDCardPrintoutComponent);

//# sourceMappingURL=IDCardPrintout.js.map