"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.BunyanInstrumentation = void 0;
const api_1 = require("@opentelemetry/api");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const version_1 = require("./version");
class BunyanInstrumentation extends instrumentation_1.InstrumentationBase {
    constructor(config = {}) {
        super('@opentelemetry/instrumentation-bunyan', version_1.VERSION, config);
    }
    init() {
        return [
            new instrumentation_1.InstrumentationNodeModuleDefinition('bunyan', ['<2.0'], logger => {
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                const proto = logger.prototype;
                if ((0, instrumentation_1.isWrapped)(proto['_emit'])) {
                    this._unwrap(proto, '_emit');
                }
                this._wrap(proto, '_emit', 
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this._getPatchedEmit());
                return logger;
            }, logger => {
                if (logger === undefined)
                    return;
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this._unwrap(logger.prototype, '_emit');
            }),
        ];
    }
    getConfig() {
        return this._config;
    }
    setConfig(config) {
        this._config = config;
    }
    _getPatchedEmit() {
        return (original) => {
            const instrumentation = this;
            return function patchedEmit(...args) {
                const span = api_1.trace.getSpan(api_1.context.active());
                if (!span) {
                    return original.apply(this, args);
                }
                const spanContext = span.spanContext();
                if (!(0, api_1.isSpanContextValid)(spanContext)) {
                    return original.apply(this, args);
                }
                const record = args[0];
                record['trace_id'] = spanContext.traceId;
                record['span_id'] = spanContext.spanId;
                record['trace_flags'] = `0${spanContext.traceFlags.toString(16)}`;
                instrumentation._callHook(span, record);
                return original.apply(this, args);
            };
        };
    }
    _callHook(span, record) {
        const hook = this.getConfig().logHook;
        if (typeof hook !== 'function') {
            return;
        }
        (0, instrumentation_1.safeExecuteInTheMiddle)(() => hook(span, record), err => {
            if (err) {
                api_1.diag.error('bunyan instrumentation: error calling logHook', err);
            }
        }, true);
    }
}
exports.BunyanInstrumentation = BunyanInstrumentation;
//# sourceMappingURL=instrumentation.js.map