"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMetadata = exports.makeGrpcClientRemoteCall = void 0;
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const api_1 = require("@opentelemetry/api");
const utils_1 = require("../utils");
const AttributeNames_1 = require("../enums/AttributeNames");
const status_code_1 = require("../status-code");
/**
 * This method handles the client remote call
 */
const makeGrpcClientRemoteCall = function (metadataCapture, original, args, metadata, self) {
    /**
     * Patches a callback so that the current span for this trace is also ended
     * when the callback is invoked.
     */
    function patchedCallback(span, callback, _metadata) {
        const wrappedFn = (err, res) => {
            if (err) {
                if (err.code) {
                    span.setStatus((0, utils_1._grpcStatusCodeToSpanStatus)(err.code));
                    span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, err.code);
                }
                span.setAttributes({
                    [AttributeNames_1.AttributeNames.GRPC_ERROR_NAME]: err.name,
                    [AttributeNames_1.AttributeNames.GRPC_ERROR_MESSAGE]: err.message,
                });
            }
            else {
                span.setStatus({ code: api_1.SpanStatusCode.UNSET });
                span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, status_code_1.GRPC_STATUS_CODE_OK);
            }
            span.end();
            callback(err, res);
        };
        return api_1.context.bind(api_1.context.active(), wrappedFn);
    }
    return (span) => {
        if (!span) {
            return original.apply(self, args);
        }
        // if unary or clientStream
        if (!original.responseStream) {
            const callbackFuncIndex = (0, utils_1.findIndex)(args, arg => {
                return typeof arg === 'function';
            });
            if (callbackFuncIndex !== -1) {
                args[callbackFuncIndex] = patchedCallback(span, args[callbackFuncIndex], metadata);
            }
        }
        span.addEvent('sent');
        setSpanContext(metadata);
        const call = original.apply(self, args);
        call.on('metadata', responseMetadata => {
            metadataCapture.client.captureResponseMetadata(span, responseMetadata);
        });
        // if server stream or bidi
        if (original.responseStream) {
            // Both error and status events can be emitted
            // the first one emitted set spanEnded to true
            let spanEnded = false;
            const endSpan = () => {
                if (!spanEnded) {
                    span.end();
                    spanEnded = true;
                }
            };
            api_1.context.bind(api_1.context.active(), call);
            call.on('error', (err) => {
                span.setStatus({
                    code: (0, utils_1._grpcStatusCodeToOpenTelemetryStatusCode)(err.code),
                    message: err.message,
                });
                span.setAttributes({
                    [AttributeNames_1.AttributeNames.GRPC_ERROR_NAME]: err.name,
                    [AttributeNames_1.AttributeNames.GRPC_ERROR_MESSAGE]: err.message,
                });
                if (err.code != null) {
                    span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, err.code);
                }
                endSpan();
            });
            call.on('status', (status) => {
                span.setStatus({ code: api_1.SpanStatusCode.UNSET });
                span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, status.code);
                endSpan();
            });
        }
        return call;
    };
};
exports.makeGrpcClientRemoteCall = makeGrpcClientRemoteCall;
const getMetadata = function (grpcClient, original, args) {
    let metadata;
    // This finds an instance of Metadata among the arguments.
    // A possible issue that could occur is if the 'options' parameter from
    // the user contains an '_internal_repr' as well as a 'getMap' function,
    // but this is an extremely rare case.
    let metadataIndex = (0, utils_1.findIndex)(args, (arg) => {
        return (arg &&
            typeof arg === 'object' &&
            arg._internal_repr &&
            typeof arg.getMap === 'function');
    });
    if (metadataIndex === -1) {
        metadata = new grpcClient.Metadata();
        if (!original.requestStream) {
            // unary or server stream
            if (args.length === 0) {
                // No argument (for the gRPC call) was provided, so we will have to
                // provide one, since metadata cannot be the first argument.
                // The internal representation of argument defaults to undefined
                // in its non-presence.
                // Note that we can't pass null instead of undefined because the
                // serializer within gRPC doesn't accept it.
                args.push(undefined);
            }
            metadataIndex = 1;
        }
        else {
            // client stream or bidi
            metadataIndex = 0;
        }
        args.splice(metadataIndex, 0, metadata);
    }
    else {
        metadata = args[metadataIndex];
    }
    return metadata;
};
exports.getMetadata = getMetadata;
const setSpanContext = function (metadata) {
    api_1.propagation.inject(api_1.context.active(), metadata, {
        set: (metadata, k, v) => metadata.set(k, v),
    });
};
//# sourceMappingURL=clientUtils.js.map