"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "importProgramRegistry", {
    enumerable: true,
    get: function() {
        return importProgramRegistry;
    }
});
const _sequelize = require("sequelize");
const _xlsx = require("xlsx");
const _logging = require("@tamanu/shared/services/logging");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _importRows = require("../importRows");
const _readMetadata = require("./readMetadata");
function readProgramRegistryData(workbook) {
    _logging.log.debug('Reading Registry data');
    const { primaryRecord: registryRecord, secondaryRecords: clinicalStatuses } = (0, _readMetadata.readTwoModelTypeSheet)(workbook.Sheets.Registry, 'Registry');
    if (!registryRecord.registryCode) {
        throw new _errors.DataImportError('Registry', -2, 'A registry must have a code');
    }
    if (!registryRecord.registryName) {
        throw new _errors.DataImportError('Registry', -2, 'A registry must have a name');
    }
    return {
        registryRecord,
        clinicalStatuses
    };
}
function readProgramRegistryConditionData(workbook) {
    _logging.log.debug('Reading Registry Condition data');
    const worksheet = workbook.Sheets['Registry Conditions'];
    if (!worksheet) {
        _logging.log.debug('No Registry Conditions sheet - skipping');
        return [];
    }
    return _xlsx.utils.sheet_to_json(worksheet);
}
const ensureUniqueName = async (context, registryName, registryId)=>{
    const conflictingRegistry = await context.models.ProgramRegistry.findOne({
        where: {
            name: registryName,
            id: {
                [_sequelize.Op.ne]: registryId
            },
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    if (conflictingRegistry) {
        throw new _errors.DataImportError('Registry', -2, `A registry name must be unique (name: ${registryName}, conflicting code: ${conflictingRegistry.code})`);
    }
};
const ensureCurrentlyAtUpdateIsAllowed = async (context, currentlyAtType, registryId)=>{
    const existingRegistry = await context.models.ProgramRegistry.findByPk(registryId);
    // No validation on first import
    if (!existingRegistry) return;
    // No validation if we aren't trying to change the currentlyAtType
    if (!currentlyAtType || currentlyAtType === existingRegistry.currentlyAtType) return;
    const existingData = await context.models.PatientProgramRegistration.findOne({
        where: {
            programRegistryId: registryId,
            [_sequelize.Op.or]: {
                facilityId: {
                    [_sequelize.Op.not]: null
                },
                villageId: {
                    [_sequelize.Op.not]: null
                }
            }
        }
    });
    if (existingData) {
        throw new _errors.DataImportError('Registry', -2, `Cannot update the currentlyAtType of a program registry with existing data`);
    }
};
async function importProgramRegistry(context, workbook, programId) {
    // There won't always be a program registry - that's fine
    _logging.log.debug('Checking for Registry sheet');
    if (!workbook.Sheets.Registry) return {};
    const { registryRecord, clinicalStatuses } = readProgramRegistryData(workbook);
    const { registryName, currentlyAtType } = registryRecord;
    const registryId = `programRegistry-${registryRecord.registryCode}`;
    await ensureUniqueName(context, registryName, registryId);
    await ensureCurrentlyAtUpdateIsAllowed(context, currentlyAtType, registryId);
    _logging.log.debug('Importing Program Registry');
    let stats = await (0, _importRows.importRows)(context, {
        sheetName: 'Registry',
        rows: [
            {
                model: 'ProgramRegistry',
                sheetRow: -2,
                values: {
                    id: registryId,
                    programId,
                    name: registryRecord.registryName,
                    code: registryRecord.registryCode,
                    visibilityStatus: registryRecord.visibilityStatus,
                    currentlyAtType: registryRecord.currentlyAtType
                }
            }
        ]
    });
    _logging.log.debug('Importing Patient Registry Clinical statuses');
    stats = await (0, _importRows.importRows)(context, {
        sheetName: 'Registry',
        rows: clinicalStatuses.map((row)=>({
                model: 'ProgramRegistryClinicalStatus',
                // Note: __rowNum__ is a non-enumerable property, so needs to be accessed explicitly here
                // -1 as it'll have 2 added to it later but it's only 1 off
                sheetRow: row.__rowNum__ - 1,
                values: {
                    id: `prClinicalStatus-${row.code}`,
                    programRegistryId: registryId,
                    ...row
                }
            })),
        stats
    });
    const programRegistryConditions = readProgramRegistryConditionData(workbook);
    _logging.log.debug('Importing Patient Registry Conditions');
    return (0, _importRows.importRows)(context, {
        sheetName: 'Registry Conditions',
        rows: programRegistryConditions.map((row)=>({
                model: 'ProgramRegistryCondition',
                // Note: __rowNum__ is a non-enumerable property, so needs to be accessed explicitly here
                // -1 as it'll have 2 added to it later but it's only 1 off
                sheetRow: row.__rowNum__ - 1,
                values: {
                    id: `prCondition-${row.code}`,
                    programRegistryId: registryId,
                    ...row
                }
            })),
        stats
    });
}

//# sourceMappingURL=importProgramRegistry.js.map