"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    getCachedAnswer: function() {
        return getCachedAnswer;
    },
    getFormDataElements: function() {
        return getFormDataElements;
    },
    getPatientById: function() {
        return getPatientById;
    },
    getPerPatientPerSurveyPerDatePerElementKey: function() {
        return getPerPatientPerSurveyPerDatePerElementKey;
    },
    getReferralDataElements: function() {
        return getReferralDataElements;
    },
    getSurveyGroupKey: function() {
        return getSurveyGroupKey;
    },
    getSurveyIdsByGroupKey: function() {
        return getSurveyIdsByGroupKey;
    },
    parametersToAnswerSqlWhere: function() {
        return parametersToAnswerSqlWhere;
    },
    removeDuplicatedAnswersPerDate: function() {
        return removeDuplicatedAnswersPerDate;
    },
    removeDuplicatedReferralsPerDate: function() {
        return removeDuplicatedReferralsPerDate;
    },
    transformAndRemoveDuplicatedAnswersPerDate: function() {
        return transformAndRemoveDuplicatedAnswersPerDate;
    }
});
const _datefns = require("date-fns");
const _lodash = require("lodash");
const _sequelize = require("sequelize");
const _dateTime = require("../../utils/dateTime");
const _utilities = require("../utilities");
const _constants = require("./constants");
const parametersToAnswerSqlWhere = (parameters)=>{
    const where = {
        '$surveyResponse.survey_id$': {
            [_sequelize.Op.in]: _constants.ALL_SURVEY_IDS
        }
    };
    if (!parameters || !Object.keys(parameters).length) {
        return where;
    }
    if (parameters.fromDate || parameters.toDate) {
        where['$surveyResponse.end_time$'] = {};
    }
    if (parameters.fromDate) {
        where['$surveyResponse.end_time$'][_sequelize.Op.gte] = (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)((0, _datefns.parseISO)(parameters.fromDate)));
    }
    if (parameters.toDate) {
        where['$surveyResponse.end_time$'][_sequelize.Op.lte] = (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(parameters.toDate)));
    }
    if (parameters.surveyId) {
        delete where['$surveyResponse.survey_id$'][_sequelize.Op.in];
        const surveyGroupKey = getSurveyGroupKey(parameters.surveyId);
        where['$surveyResponse.survey_id$'][_sequelize.Op.in] = getSurveyIdsByGroupKey(surveyGroupKey);
    }
    return where;
};
const getPatients = async (models, patientIds)=>models.Patient.findAll({
        include: [
            {
                model: models.PatientAdditionalData,
                as: 'additionalData',
                include: [
                    'ethnicity',
                    'medicalArea',
                    'nursingZone'
                ]
            },
            'village'
        ],
        where: {
            id: {
                [_sequelize.Op.in]: patientIds
            }
        },
        order: [
            [
                'firstName',
                'ASC'
            ]
        ]
    });
const getPerPatientPerSurveyPerDatePerElementKey = (patientId, surveyGroupKey, responseDate, dataElementId)=>`${patientId}|${surveyGroupKey}|${responseDate}|${dataElementId}`;
const getCachedAnswer = (answersByPatientSurveyDataElement, patientId, surveyGroupKey, responseDate, dataElementId)=>answersByPatientSurveyDataElement[getPerPatientPerSurveyPerDatePerElementKey(patientId, surveyGroupKey, responseDate, dataElementId)]?.body;
const getSurveyGroupKey = (surveyId)=>{
    if (_constants.CVD_SURVEY_IDS.includes(surveyId)) {
        return _constants.CVD_SURVEY_GROUP_KEY;
    }
    if (_constants.BREAST_CANCER_SURVEY_IDS.includes(surveyId)) {
        return _constants.BREAST_CANCER_SURVEY_GROUP_KEY;
    }
    if (_constants.CERVICAL_CANCER_SURVEY_IDS.includes(surveyId)) {
        return _constants.CERVICAL_CANCER_SURVEY_GROUP_KEY;
    }
    // should never happen
    throw new Error(`Unknown survey id: ${surveyId}`);
};
const getSurveyIdsByGroupKey = (surveyGroupKey)=>{
    switch(surveyGroupKey){
        case _constants.BREAST_CANCER_SURVEY_GROUP_KEY:
            return _constants.BREAST_CANCER_SURVEY_IDS;
        case _constants.CVD_SURVEY_GROUP_KEY:
            return _constants.CVD_SURVEY_IDS;
        case _constants.CERVICAL_CANCER_SURVEY_GROUP_KEY:
        default:
            return _constants.CERVICAL_CANCER_SURVEY_IDS;
    }
};
const getFormDataElements = (surveyGroupKey)=>{
    switch(surveyGroupKey){
        case _constants.CVD_SURVEY_GROUP_KEY:
            return _constants.CVD_PRIMARY_SCREENING_FORM_DATA_ELEMENT_IDS;
        case _constants.BREAST_CANCER_SURVEY_GROUP_KEY:
            return _constants.BREAST_CANCER_PRIMARY_SCREENING_FORM_DATA_ELEMENT_IDS;
        case _constants.CERVICAL_CANCER_SURVEY_GROUP_KEY:
        default:
            return _constants.CERVICAL_CANCER_PRIMARY_SCREENING_FORM_DATA_ELEMENT_IDS;
    }
};
const getReferralDataElements = (surveyGroupKey)=>{
    switch(surveyGroupKey){
        case _constants.CVD_SURVEY_GROUP_KEY:
            return _constants.CVD_PRIMARY_SCREENING_REFERRAL_DATA_ELEMENT_IDS;
        case _constants.BREAST_CANCER_SURVEY_GROUP_KEY:
            return _constants.BREAST_CANCER_PRIMARY_SCREENING_REFERRAL_DATA_ELEMENT_IDS;
        case _constants.CERVICAL_CANCER_SURVEY_GROUP_KEY:
        default:
            return _constants.CERVICAL_CANCER_PRIMARY_SCREENING_REFERRAL_DATA_ELEMENT_IDS;
    }
};
const getPatientById = async (models, rawAnswers)=>{
    const patientIds = rawAnswers.map((a)=>a.surveyResponse?.encounter?.patientId);
    const patients = await getPatients(models, patientIds);
    return (0, _lodash.keyBy)(patients, 'id');
};
const removeDuplicatedReferralsPerDate = (referrals)=>{
    const referralByPatientAndDate = (0, _lodash.groupBy)(referrals, (r)=>{
        const referralDate = (0, _dateTime.format)(r.surveyResponse.endTime, 'dd-MM-yyyy');
        return `${r.initiatingEncounter.patientId}|${r.surveyResponse.surveyId}|${referralDate}`;
    });
    const results = [];
    for (const groupedAnswers of Object.values(referralByPatientAndDate)){
        const sortedLatestToOldestReferrals = groupedAnswers.sort((r1, r2)=>(0, _dateTime.differenceInMilliseconds)(r2.initiatingEncounter.startDate, r1.initiatingEncounter.startDate));
        results.push(sortedLatestToOldestReferrals[0]);
    }
    return results;
};
const removeDuplicatedAnswersPerDate = (answers)=>{
    const answersPerElement = (0, _lodash.groupBy)(answers, (a)=>{
        const responseDate = (0, _dateTime.format)(a.responseEndTime, 'dd-MM-yyyy');
        return `${a.patientId}|${a.surveyId}|${responseDate}|${a.dataElementId}`;
    });
    const results = [];
    for (const groupedAnswers of Object.values(answersPerElement)){
        const sortedLatestToOldestAnswers = groupedAnswers.sort((a1, a2)=>(0, _dateTime.differenceInMilliseconds)(a2.responseEndTime, a1.responseEndTime));
        results.push(sortedLatestToOldestAnswers[0]);
    }
    return results;
};
const transformAndRemoveDuplicatedAnswersPerDate = async (models, rawAnswers, surveyIds)=>{
    const components = await models.SurveyScreenComponent.getComponentsForSurveys(surveyIds, {
        includeAllVitals: true
    });
    const answersIncludingResults = [
        ...rawAnswers,
        ...getSurveyResultsFromAnswers(rawAnswers)
    ];
    const transformedAnswers = await (0, _utilities.transformAnswers)(models, answersIncludingResults, components);
    return removeDuplicatedAnswersPerDate(transformedAnswers);
};
const getSurveyResultBody = (surveyResponse)=>{
    // There is also a `result` in surveyResponse, but since
    // it is not used for fiji-ncd and this is a local util,
    // we just use resultText.
    const { surveyId, resultText } = surveyResponse;
    if (surveyId === _constants.BREAST_CANCER_FORM_SURVEY_ID) {
        return resultText || 'Not high risk';
    }
    return resultText;
};
const getSurveyResultsFromAnswers = (answers)=>{
    const surveyResponses = answers.map((a)=>a.surveyResponse);
    const seenSurveyResponseIds = new Set();
    const uniqueSurveyResponses = surveyResponses.filter(({ id })=>{
        if (seenSurveyResponseIds.has(id)) {
            return false;
        }
        seenSurveyResponseIds.add(id);
        return true;
    });
    const surveyResults = uniqueSurveyResponses.map((sr)=>({
            dataElementId: (0, _constants.getSurveyResultDataElement)(sr.surveyId),
            surveyResponse: sr,
            body: getSurveyResultBody(sr)
        }));
    return surveyResults;
};

//# sourceMappingURL=utils.js.map