"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const api_1 = require("@opentelemetry/api");
const assert = require("assert");
const sinon = require("sinon");
const src_1 = require("../src");
describe('utils', () => {
    describe('getNodeAutoInstrumentations', () => {
        it('should include all installed instrumentations', () => {
            const instrumentations = (0, src_1.getNodeAutoInstrumentations)();
            const installedInstrumentations = Object.keys(require('../package.json').dependencies).filter(depName => {
                return depName.startsWith('@opentelemetry/instrumentation-');
            });
            assert.deepStrictEqual(new Set(instrumentations.map(i => i.instrumentationName)), new Set(installedInstrumentations));
        });
        it('should use user config', () => {
            function applyCustomAttributesOnSpan() { }
            const instrumentations = (0, src_1.getNodeAutoInstrumentations)({
                '@opentelemetry/instrumentation-http': {
                    applyCustomAttributesOnSpan,
                },
            });
            const instrumentation = instrumentations.find(instr => instr.instrumentationName === '@opentelemetry/instrumentation-http');
            const configHttp = instrumentation._config;
            assert.strictEqual(configHttp.applyCustomAttributesOnSpan, applyCustomAttributesOnSpan);
        });
        it('should not return disabled instrumentation', () => {
            const instrumentations = (0, src_1.getNodeAutoInstrumentations)({
                '@opentelemetry/instrumentation-grpc': {
                    enabled: false,
                },
            });
            const instrumentation = instrumentations.find(instr => instr.instrumentationName === '@opentelemetry/instrumentation-grpc');
            assert.strictEqual(instrumentation, undefined);
        });
        it('should show error for none existing instrumentation', () => {
            const spy = sinon.stub(api_1.diag, 'error');
            const name = '@opentelemetry/instrumentation-http2';
            const instrumentations = (0, src_1.getNodeAutoInstrumentations)({
                // @ts-expect-error verify that wrong name works
                [name]: {
                    enabled: false,
                },
            });
            const instrumentation = instrumentations.find(instr => instr.instrumentationName === name);
            assert.strictEqual(instrumentation, undefined);
            assert.strictEqual(spy.args[0][0], `Provided instrumentation name "${name}" not found`);
        });
    });
});
//# sourceMappingURL=utils.test.js.map