"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    HCERTPack: function() {
        return HCERTPack;
    },
    HCERTVerify: function() {
        return HCERTVerify;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _cbor = /*#__PURE__*/ _interop_require_default(require("cbor"));
const _cosejs = /*#__PURE__*/ _interop_require_default(require("cose-js"));
const _logging = require("@tamanu/shared/services/logging");
const _asn1js = require("asn1js");
const _zlib = require("zlib");
const _util = require("util");
const _base45js = /*#__PURE__*/ _interop_require_default(require("base45-js"));
const _pkijs = require("pkijs");
const _utils = require("@tamanu/shared/utils");
const _datefns = require("date-fns");
const _Signer = require("../Signer");
const _localisation = require("../../localisation");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const deflate = (0, _util.promisify)(_zlib.deflate);
const inflate = (0, _util.promisify)(_zlib.inflate);
const EUDGC_IN_HCERT_KEY = 1;
const CWT_CLAIM_KEYS = {
    iss: 1,
    exp: 4,
    iat: 6,
    hcert: -260
};
/**
 *  Fetches the actual 32 byte privateKey from within the structured privateKey data
 *
 *  @returns 32-byte buffer with the privateKey value
 */ function extractKeyD(keyData) {
    const asn = (0, _asn1js.fromBER)((0, _Signer.fakeABtoRealAB)(keyData.export({
        type: 'pkcs8',
        format: 'der'
    }).buffer));
    if (asn.result.error !== '') {
        throw new Error(asn.result.error);
    }
    const [, , octetString] = asn.result.valueBlock.value;
    const [octetSequence] = octetString.valueBlock.value;
    const [, privateKey] = octetSequence.valueBlock.value;
    if (privateKey.valueBlock.blockLength !== 32) {
        throw new Error(`Private key block length ${privateKey.valueBlock.blockLength} instead of 32`);
    }
    return Buffer.from(privateKey.valueBlock.valueHex, 'hex');
}
async function HCERTPack(messageData, { models }) {
    _logging.log.info('HCERT Packing message data');
    const signer = await models.Signer.findActive();
    if (!signer) {
        throw new Error('Cannot pack HCERT, no active signer');
    }
    const iss = (await (0, _localisation.getLocalisation)()).country['alpha-2'];
    const iat = new Date();
    const exp = (0, _datefns.add)(iat, {
        days: 365
    });
    const hcert = new Map();
    hcert.set(EUDGC_IN_HCERT_KEY, messageData);
    const payload = new Map();
    payload.set(CWT_CLAIM_KEYS.iss, iss);
    payload.set(CWT_CLAIM_KEYS.iat, (0, _datefns.getUnixTime)(iat));
    payload.set(CWT_CLAIM_KEYS.exp, (0, _datefns.getUnixTime)(exp));
    payload.set(CWT_CLAIM_KEYS.hcert, hcert);
    const cborData = _cbor.default.encode(payload);
    // p - protected
    // u - unprotected
    const coseHeaders = {
        p: {
            alg: 'ES256',
            kid: signer.id
        },
        u: {}
    };
    const coseSigner = {
        key: {
            d: extractKeyD(signer.decryptPrivateKey(_config.default.integrations.signer.keySecret))
        }
    };
    const signedData = await _cosejs.default.sign.create(coseHeaders, cborData, coseSigner);
    await signer.increment('signaturesIssued');
    const deflatedBuf = await deflate(signedData);
    return `HC1:${_base45js.default.encode(deflatedBuf)}`;
}
async function HCERTVerify(packedData, { models }) {
    _logging.log.info('Verifying HCERT message');
    const signer = await models.Signer.findActive();
    // Fetch publicKey data from cert
    // Parsing the publicKey field directly seems to go wonky
    const cert = (0, _utils.depem)(signer.certificate, 'CERTIFICATE');
    const asn = (0, _asn1js.fromBER)((0, _Signer.fakeABtoRealAB)(cert));
    const certificate = new _pkijs.Certificate({
        schema: asn.result
    });
    const verifier = {
        key: {
            x: Buffer.from(certificate.subjectPublicKeyInfo.parsedKey.x),
            y: Buffer.from(certificate.subjectPublicKeyInfo.parsedKey.y),
            kid: signer.id
        }
    };
    // Strip HC1: header
    if (!packedData.startsWith('HC1:')) {
        _logging.log.error('No HC1 header detected in HCERT data');
        throw new Error('No HC1 header detected in HCERT data');
    }
    const strippedData = packedData.substring(4);
    const decodedData = _base45js.default.decode(strippedData);
    const inflatedData = await inflate(decodedData);
    const verifiedData = await _cosejs.default.sign.verify(inflatedData, verifier);
    const payload = _cbor.default.decode(verifiedData);
    if (payload.get(CWT_CLAIM_KEYS.iss) !== (await (0, _localisation.getLocalisation)()).country['alpha-2']) {
        throw new Error('HCERT message issued to wrong country');
    }
    const now = (0, _datefns.getUnixTime)(new Date());
    if (payload.get(CWT_CLAIM_KEYS.iat) > now) {
        throw new Error('HCERT message issued in the future');
    }
    if (payload.get(CWT_CLAIM_KEYS.exp) < now) {
        throw new Error('HCERT message expired');
    }
    const hcert = payload.get(CWT_CLAIM_KEYS.hcert);
    if (!hcert) {
        throw new Error('HCERT message missing hcert claim');
    }
    if (!hcert.has(EUDGC_IN_HCERT_KEY)) {
        throw new Error('HCERT message missing EUDGC claim');
    }
    return hcert.get(EUDGC_IN_HCERT_KEY);
}

//# sourceMappingURL=Crypto.js.map