"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    generateInitialVitalLogs: function() {
        return generateInitialVitalLogs;
    },
    generateInitialVitalLogsCommand: function() {
        return generateInitialVitalLogsCommand;
    }
});
const _commander = require("commander");
const _logging = require("@tamanu/shared/services/logging");
const _constants = require("@tamanu/constants");
const _dateTime = require("@tamanu/shared/utils/dateTime");
const _database = require("../database");
const SURVEY_RESPONSE_BATCH_SIZE = 1000;
async function generateVitalLogsInBatch(store, vitalsSurveyId, batchSize, offset) {
    const { sequelize } = store;
    const [, metadata] = await sequelize.query(`
      WITH
        paginated_survey_responses AS (
          SELECT id, end_time, start_time, user_id
          FROM survey_responses
          WHERE survey_id = :vitalsSurveyId
          AND deleted_at is null
          ORDER BY created_at ASC, id ASC
          LIMIT :limit
          OFFSET :offset
        )

      INSERT INTO
        vital_logs (created_at, updated_at, date, new_value, recorded_by_id, answer_id)
      SELECT
        now() as created_at,
        now() as updated_at,
        COALESCE(psr.end_time, psr.start_time, :currentDateTimeString) as date,
        sra.body as new_value,
        psr.user_id as recorded_by_id,
        sra.id as answer_id
      FROM
        paginated_survey_responses psr
      INNER JOIN
        survey_response_answers sra ON psr.id = sra.response_id
        AND
          sra.body IS NOT NULL
        AND
          sra.body != ''
        AND
          sra.data_element_id != :dateDataElementId
      LEFT JOIN
        vital_logs vl ON vl.answer_id = sra.id
      WHERE
        vl.id IS NULL;
    `, {
        replacements: {
            vitalsSurveyId,
            limit: batchSize,
            offset,
            dateDataElementId: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded,
            currentDateTimeString: (0, _dateTime.getCurrentDateTimeString)()
        }
    });
    return metadata.rowCount;
}
async function generateInitialVitalLogs() {
    const store = await (0, _database.initDatabase)({
        testMode: false
    });
    const { Survey, SurveyResponse } = store.models;
    try {
        const vitalsSurvey = await Survey.getVitalsSurvey();
        const surveyResponseCount = await SurveyResponse.count({
            where: {
                surveyId: vitalsSurvey.id
            }
        });
        const batchCount = Math.ceil(surveyResponseCount / SURVEY_RESPONSE_BATCH_SIZE);
        _logging.log.info(`Generating initial vital logs for ${surveyResponseCount} survey responses in ${batchCount} batches of ${SURVEY_RESPONSE_BATCH_SIZE}`);
        let totalCreated = 0;
        // Run in batches to avoid OOM issue
        for(let i = 0; i < batchCount; i++){
            _logging.log.info(`Generating initial vital logs for batch ${i + 1}/${batchCount}`);
            const logsCreatedInBatch = await generateVitalLogsInBatch(store, vitalsSurvey.id, SURVEY_RESPONSE_BATCH_SIZE, i * SURVEY_RESPONSE_BATCH_SIZE);
            totalCreated += logsCreatedInBatch;
        }
        _logging.log.info(`Successfully created ${totalCreated} initial vital logs.`);
        process.exit(0);
    } catch (error) {
        _logging.log.info(`Command failed: ${error.stack}\n`);
        process.exit(1);
    }
}
const generateInitialVitalLogsCommand = new _commander.Command('generateInitialVitalLogs').description('Generates missing initial vital log records for survey response answers').action(generateInitialVitalLogs);

//# sourceMappingURL=generateInitialVitalLogs.js.map