"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "CertificateNotificationProcessor", {
    enumerable: true,
    get: function() {
        return CertificateNotificationProcessor;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _lodash = require("lodash");
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _tasks = require("@tamanu/shared/tasks");
const _uvci = require("@tamanu/shared/utils/uvci");
const _patientCertificates = require("@tamanu/shared/utils/patientCertificates");
const _makePatientCertificate = require("../../utils/makePatientCertificate");
const _localisation = require("../../localisation");
const _VdsNc = require("../../integrations/VdsNc");
const _EuDcc = require("../../integrations/EuDcc");
const _LabRequestNotificationGenerator = require("./LabRequestNotificationGenerator");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let CertificateNotificationProcessor = class CertificateNotificationProcessor extends _tasks.ScheduledTask {
    getName() {
        return 'CertificateNotificationProcessor';
    }
    async countQueue() {
        return this.context.store.models.CertificateNotification.count({
            where: {
                status: _constants.CERTIFICATE_NOTIFICATION_STATUSES.QUEUED
            }
        });
    }
    async run() {
        const { models, sequelize } = this.context.store;
        const { CertificateNotification, CertifiableVaccine, PatientCommunication, Patient, TranslatedString } = models;
        const vdsEnabled = _config.default.integrations.vdsNc.enabled;
        const euDccEnabled = _config.default.integrations.euDcc.enabled;
        const localisation = await (0, _localisation.getLocalisation)();
        const certifiableVaccineIds = await CertifiableVaccine.allVaccineIds(euDccEnabled);
        const queuedNotifications = await CertificateNotification.findAll({
            where: {
                status: _constants.CERTIFICATE_NOTIFICATION_STATUSES.QUEUED
            },
            order: [
                [
                    'createdAt',
                    'ASC'
                ]
            ],
            limit: this.config.limit
        });
        let processed = 0;
        for (const notification of queuedNotifications){
            try {
                const patientId = notification.get('patientId');
                const patient = await Patient.findByPk(patientId);
                const requireSigning = notification.get('requireSigning');
                const type = notification.get('type');
                const printedBy = notification.get('createdBy');
                const printedDate = notification.get('printedDate');
                const facilityName = notification.get('facilityName');
                const language = notification.get('language');
                const translations = await TranslatedString.getTranslations(language, [
                    'pdf'
                ]);
                const { country } = await (0, _localisation.getLocalisation)();
                const countryCode = country['alpha-2'];
                const sublog = _logging.log.child({
                    id: notification.id,
                    patient: patientId,
                    type,
                    requireSigning
                });
                sublog.info('Processing certificate notification');
                let template;
                let qrData = null;
                let pdf = null;
                switch(type){
                    case _constants.ICAO_DOCUMENT_TYPES.PROOF_OF_VACCINATION.JSON:
                        {
                            template = 'covidVaccineCertificateEmail';
                            const latestCertifiableVax = await models.AdministeredVaccine.lastVaccinationForPatient(patient.id, certifiableVaccineIds);
                            let uvci;
                            if (requireSigning && latestCertifiableVax) {
                                if (euDccEnabled) {
                                    sublog.debug('Generating EU DCC data for proof of vaccination', {
                                        vax: latestCertifiableVax.id
                                    });
                                    uvci = await (0, _uvci.generateUVCI)(latestCertifiableVax.id, {
                                        format: 'eudcc',
                                        countryCode
                                    });
                                    const povData = await (0, _EuDcc.createEuDccVaccinationData)(latestCertifiableVax.id, {
                                        models
                                    });
                                    qrData = await (0, _EuDcc.HCERTPack)(povData, {
                                        models
                                    });
                                } else if (vdsEnabled) {
                                    sublog.debug('Generating VDS data for proof of vaccination', {
                                        vax: latestCertifiableVax.id
                                    });
                                    uvci = await (0, _uvci.generateUVCI)(latestCertifiableVax.id, {
                                        format: 'icao',
                                        countryCode
                                    });
                                    const povData = await (0, _VdsNc.createVdsNcVaccinationData)(patient.id, {
                                        models
                                    });
                                    const vdsDoc = new _VdsNc.VdsNcDocument(type, povData, uvci);
                                    vdsDoc.models = models;
                                    await vdsDoc.sign();
                                    qrData = await vdsDoc.intoVDS();
                                } else if (requireSigning) {
                                    sublog.warn('Signing is required but certificate contains no certifiable vaccines');
                                } else {
                                    sublog.error('Signing is required but neither EU DCC nor VDS is enabled');
                                }
                            }
                            sublog.info('Generating vax certificate PDF', {
                                uvci
                            });
                            pdf = await (0, _makePatientCertificate.makeCovidVaccineCertificate)(patient, printedBy, printedDate, models, uvci, qrData, language);
                            break;
                        }
                    case _constants.ICAO_DOCUMENT_TYPES.PROOF_OF_TESTING.JSON:
                        {
                            // let uvci;
                            template = 'covidTestCertificateEmail';
                            if (requireSigning && vdsEnabled) {
                            // sublog.debug('Generating VDS data for proof of testing');
                            // uvci = await generateUVCI(latestCovidVax.id, { format: 'icao', countryCode });
                            // const povData = await createVdsNcTestData(patient.id, { models });
                            // const vdsDoc = new VdsNcDocument(type, povData, uvci);
                            // vdsDoc.models = models;
                            // await vdsDoc.sign();
                            // qrData = await vdsDoc.intoVDS();
                            }
                            sublog.info('Generating test certificate PDF');
                            pdf = await (0, _makePatientCertificate.makeCovidCertificate)(_patientCertificates.CertificateTypes.test, patient, printedBy, models, qrData, language);
                            break;
                        }
                    case _constants.COVID_19_CLEARANCE_CERTIFICATE:
                        template = 'covidClearanceCertificateEmail';
                        sublog.info('Generating clearance certificate PDF');
                        pdf = await (0, _makePatientCertificate.makeCovidCertificate)(_patientCertificates.CertificateTypes.clearance, patient, printedBy, models, qrData, language);
                        break;
                    case _constants.VACCINATION_CERTIFICATE:
                        template = 'vaccineCertificateEmail';
                        pdf = await (0, _makePatientCertificate.makeVaccineCertificate)(patient, printedBy, printedDate, facilityName, models, language, translations);
                        break;
                    default:
                        throw new Error(`Unknown certificate type ${type}`);
                }
                sublog.debug('Creating communication record');
                // eslint-disable-next-line no-loop-func
                const [comm] = await sequelize.transaction(()=>// queue the email to be sent and mark this notification as processed
                    Promise.all([
                        PatientCommunication.create({
                            type: _constants.PATIENT_COMMUNICATION_TYPES.CERTIFICATE,
                            channel: _constants.PATIENT_COMMUNICATION_CHANNELS.EMAIL,
                            subject: (0, _lodash.get)(localisation, `templates.${template}.subject`),
                            content: (0, _lodash.get)(localisation, `templates.${template}.body`),
                            status: _constants.COMMUNICATION_STATUSES.QUEUED,
                            patientId,
                            destination: notification.get('forwardAddress'),
                            attachment: pdf.filePath
                        }),
                        notification.update({
                            status: _constants.CERTIFICATE_NOTIFICATION_STATUSES.PROCESSED
                        })
                    ]));
                processed += 1;
                sublog.info('Done processing certificate notification', {
                    emailId: comm.id
                });
            } catch (error) {
                _logging.log.error('Failed to process certificate notification', {
                    id: notification.id,
                    error
                });
                await notification.update({
                    status: _constants.CERTIFICATE_NOTIFICATION_STATUSES.ERROR,
                    error: error.message
                });
            }
        }
        _logging.log.info('Done with certificate notification task', {
            attempted: queuedNotifications.length,
            processed
        });
    }
    constructor(context){
        const conf = _config.default.schedules.certificateNotificationProcessor;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.config = conf;
        this.context = context;
        this.subtasks = [
            new _LabRequestNotificationGenerator.LabRequestNotificationGenerator(context)
        ];
    }
};

//# sourceMappingURL=index.js.map