"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "DocumentMetadata", {
    enumerable: true,
    get: function() {
        return DocumentMetadata;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _dateTimeTypes = require("./dateTimeTypes");
const _dateTime = require("../utils/dateTime");
const _buildEncounterLinkedSyncFilter = require("./buildEncounterLinkedSyncFilter");
const _onSaveMarkPatientForSync = require("./onSaveMarkPatientForSync");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
let DocumentMetadata = class DocumentMetadata extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            name: {
                type: _sequelize.Sequelize.TEXT,
                allowNull: false
            },
            type: {
                type: _sequelize.Sequelize.TEXT,
                allowNull: false
            },
            source: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false,
                defaultValue: _constants.DOCUMENT_SOURCES.UPLOADED
            },
            documentCreatedAt: (0, _dateTimeTypes.dateTimeType)('documentCreatedAt'),
            documentUploadedAt: (0, _dateTimeTypes.dateTimeType)('documentUploadedAt', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            documentOwner: _sequelize.Sequelize.TEXT,
            note: _sequelize.Sequelize.STRING,
            // Relation can't be managed by sequelize because the
            // attachment won't get downloaded to facility server
            attachmentId: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
        (0, _onSaveMarkPatientForSync.onSaveMarkPatientForSync)(this);
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.Patient, {
            foreignKey: 'patientId',
            as: 'patient'
        });
        this.belongsTo(models.Department, {
            foreignKey: 'departmentId',
            as: 'department'
        });
    }
    static getListReferenceAssociations() {
        return [
            'department'
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        const join = (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
            this.tableName,
            'encounters'
        ]);
        return `
      ${join}
      WHERE (
        encounters.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
        OR
        ${this.tableName}.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
      )
      AND ${this.tableName}.updated_at_sync_tick > :since
    `;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: 'COALESCE(document_metadata.patient_id, encounters.patient_id)'
            }),
            joins: `
        LEFT JOIN encounters ON ${this.tableName}.encounter_id = encounters.id
        LEFT JOIN patients ON ${this.tableName}.patient_id = encounters.id
      `
        };
    }
};

//# sourceMappingURL=DocumentMetadata.js.map