"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "FhirJob", {
    enumerable: true,
    get: function() {
        return FhirJob;
    }
});
const _api = require("@opentelemetry/api");
const _ms = /*#__PURE__*/ _interop_require_default(require("ms"));
const _sequelize = /*#__PURE__*/ _interop_require_wildcard(require("sequelize"));
const _constants = require("@tamanu/constants");
const _Model = require("../Model");
const _logging = require("../../services/logging");
const _sleepAsync = require("../../utils/sleepAsync");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
let FhirJob = class FhirJob extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: {
                ...primaryKey,
                type: _sequelize.DataTypes.UUID,
                defaultValue: _sequelize.default.fn('uuid_generate_v4')
            },
            // queue
            priority: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: false,
                defaultValue: 1000
            },
            status: {
                type: _sequelize.DataTypes.TEXT,
                defaultValue: 'Queued',
                allowNull: false
            },
            worker_id: _sequelize.DataTypes.UUID,
            started_at: _sequelize.DataTypes.DATE,
            completed_at: _sequelize.DataTypes.DATE,
            errored_at: _sequelize.DataTypes.DATE,
            error: _sequelize.DataTypes.TEXT,
            // routing
            topic: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            discriminant: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false,
                defaultValue: _sequelize.default.fn('uuid_generate_v4'),
                unique: true
            },
            // data
            payload: {
                type: _sequelize.DataTypes.JSONB,
                allowNull: false,
                defaultValue: {}
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.DO_NOT_SYNC,
            schema: 'fhir',
            tableName: 'jobs',
            indexes: [
                {
                    fields: [
                        'topic',
                        'status',
                        'priority',
                        'created_at'
                    ]
                }
            ]
        });
    }
    static async backlogUntilLimit(topic, limit, includeDropped = true) {
        // Retrieving the size of the whole backlog can be expensive, and sometimes
        // we only need to check how many records can be retrieved up to a limit of
        // n, so this method returns the minimum of limit and backlog size
        const [{ count }] = await this.sequelize.query(`SELECT fhir.job_backlog_until_limit($topic, $limit, $includeDropped) as count`, {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                topic,
                limit,
                includeDropped
            }
        });
        return count;
    }
    static async grab(workerId, topic) {
        // We need to have strong isolation when grabbing, to avoid grabbing the
        // same job twice. But that runs the risk of failing due to serialization
        // failures, so we retry a few times, and add a bit of jitter to increase
        // our chances of success.
        const GRAB_RETRY = 10;
        for(let i = 0; i < GRAB_RETRY; i += 1){
            try {
                return await this.sequelize.transaction(async ()=>{
                    const [{ id }] = await this.sequelize.query('SELECT (fhir.job_grab($workerId, $topic)).job_id as id', {
                        type: _sequelize.QueryTypes.SELECT,
                        bind: {
                            workerId,
                            topic
                        }
                    });
                    if (!id) return null;
                    return FhirJob.findByPk(id);
                });
            } catch (err) {
                _logging.log.debug(`Failed to grab job`, err);
                // retry, with a bit of jitter to avoid thundering herd
                const delay = Math.floor(Math.random() * 500);
                // eslint-disable-next-line no-unused-expressions
                _api.trace.getActiveSpan()?.addEvent('grab retry', {
                    delay,
                    attempt: i + 1
                });
                _logging.log.debug(`Failed to grab job, retrying in ${(0, _ms.default)(delay)} (${i + 1}/${GRAB_RETRY})`);
                if (i > GRAB_RETRY / 2) {
                    _logging.log.warn(`Failed to grab job after ${i + 1} retries, this is unusual`);
                }
                await (0, _sleepAsync.sleepAsync)(delay);
                continue;
            }
        }
        throw new Error(`Failed to grab job after ${GRAB_RETRY} retries`);
    }
    static async submit(topic, payload = {}, { priority = 1000, discriminant = null } = {}) {
        const [{ id }] = await this.sequelize.query(`
      SELECT fhir.job_submit(
          $topic
        , $payload
        , $priority
        ${discriminant ? ', $discriminant' : ''}
      ) as id
    `, {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                topic,
                payload,
                priority,
                discriminant
            }
        });
        return id;
    }
    async start(workerId) {
        await this.sequelize.query('SELECT fhir.job_start($jobId, $workerId)', {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                jobId: this.id,
                workerId
            }
        });
    }
    async complete(workerId) {
        await this.sequelize.query('SELECT fhir.job_complete($jobId, $workerId)', {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                jobId: this.id,
                workerId
            }
        });
    }
    async fail(workerId, error) {
        await this.sequelize.query('SELECT fhir.job_fail($jobId, $workerId, $error)', {
            type: _sequelize.QueryTypes.SELECT,
            bind: {
                jobId: this.id,
                workerId,
                error
            }
        });
        await this.reload();
        return this;
    }
};

//# sourceMappingURL=Job.js.map