/**
 * Encode DER data as a PEM document.
 * @param {Buffer} data DER data
 * @param {string} banner Uppercase string for the BEGIN/END banners
 * @returns {string} PEM document
 */ "use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    base64UrlDecode: function() {
        return base64UrlDecode;
    },
    base64UrlEncode: function() {
        return base64UrlEncode;
    },
    depem: function() {
        return depem;
    },
    jsonFromBase64: function() {
        return jsonFromBase64;
    },
    jsonToBase64: function() {
        return jsonToBase64;
    },
    pem: function() {
        return pem;
    }
});
function pem(data, banner) {
    return `-----BEGIN ${banner}-----\n${data.toString('base64').match(/.{1,64}/g).join('\n')}\n-----END ${banner}-----`;
}
function depem(pemString, expectedBanner) {
    const text = pemString.trim();
    const beginRx = /^-{5}\s*BEGIN ?([^-]+)?-{5}\r?\n/;
    const endRx = /\r?\n-{5}\s*END ?([^-]+)?-{5}$/;
    const beginMatch = text.match(beginRx);
    if (!beginMatch || beginMatch[1] !== expectedBanner) {
        throw new Error(`Missing start banner on PEM, expected '-----BEGIN ${expectedBanner}-----'`);
    }
    const endMatch = text.match(endRx);
    if (!endMatch || endMatch[1] !== expectedBanner) {
        throw new Error(`Missing end banner on PEM, expected '-----END ${expectedBanner}-----'`);
    }
    return Buffer.from(text.replace(/^--.+/gm, ''), 'base64');
}
function base64UrlEncode(input) {
    return Buffer.from(input).toString('base64url');
}
function base64UrlDecode(input) {
    return Buffer.from(input, 'base64url');
}
function jsonFromBase64(base64str) {
    return JSON.parse(Buffer.from(base64str, 'base64').toString('binary'));
}
function jsonToBase64(obj) {
    return Buffer.from(JSON.stringify(obj), 'binary').toString('base64');
}

//# sourceMappingURL=encodings.js.map