"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KnexInstrumentation = void 0;
const api = require("@opentelemetry/api");
const version_1 = require("./version");
const constants = require("./constants");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const utils = require("./utils");
const contextSymbol = Symbol('opentelemetry.instrumentation-knex.context');
const DEFAULT_CONFIG = {
    maxQueryLength: 1022,
};
class KnexInstrumentation extends instrumentation_1.InstrumentationBase {
    constructor(config = {}) {
        super(`@opentelemetry/instrumentation-${constants.MODULE_NAME}`, version_1.VERSION, Object.assign({}, DEFAULT_CONFIG, config));
    }
    init() {
        const module = new instrumentation_1.InstrumentationNodeModuleDefinition(constants.MODULE_NAME, constants.SUPPORTED_VERSIONS);
        module.files.push(this.getClientNodeModuleFileInstrumentation('src'), this.getClientNodeModuleFileInstrumentation('lib'), this.getRunnerNodeModuleFileInstrumentation('src'), this.getRunnerNodeModuleFileInstrumentation('lib'), this.getRunnerNodeModuleFileInstrumentation('lib/execution'));
        return module;
    }
    getRunnerNodeModuleFileInstrumentation(basePath) {
        return new instrumentation_1.InstrumentationNodeModuleFile(`knex/${basePath}/runner.js`, constants.SUPPORTED_VERSIONS, (Runner, moduleVersion) => {
            api.diag.debug(`Applying ${basePath}/runner.js patch for ${constants.MODULE_NAME}@${moduleVersion}`);
            this.ensureWrapped(moduleVersion, Runner.prototype, 'query', this.createQueryWrapper(moduleVersion));
            return Runner;
        }, (Runner, moduleVersion) => {
            api.diag.debug(`Removing ${basePath}/runner.js patch for ${constants.MODULE_NAME}@${moduleVersion}`);
            this._unwrap(Runner.prototype, 'query');
            return Runner;
        });
    }
    getClientNodeModuleFileInstrumentation(basePath) {
        return new instrumentation_1.InstrumentationNodeModuleFile(`knex/${basePath}/client.js`, constants.SUPPORTED_VERSIONS, (Client, moduleVersion) => {
            api.diag.debug(`Applying ${basePath}/client.js patch for ${constants.MODULE_NAME}@${moduleVersion}`);
            this.ensureWrapped(moduleVersion, Client.prototype, 'queryBuilder', this.storeContext.bind(this));
            this.ensureWrapped(moduleVersion, Client.prototype, 'schemaBuilder', this.storeContext.bind(this));
            this.ensureWrapped(moduleVersion, Client.prototype, 'raw', this.storeContext.bind(this));
            return Client;
        }, (Client, moduleVersion) => {
            api.diag.debug(`Removing ${basePath}/client.js patch for ${constants.MODULE_NAME}@${moduleVersion}`);
            this._unwrap(Client.prototype, 'queryBuilder');
            this._unwrap(Client.prototype, 'schemaBuilder');
            this._unwrap(Client.prototype, 'raw');
            return Client;
        });
    }
    createQueryWrapper(moduleVersion) {
        const instrumentation = this;
        return function wrapQuery(original) {
            return function wrapped_logging_method(query) {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const config = this.client.config;
                const table = (_b = (_a = this.builder) === null || _a === void 0 ? void 0 : _a._single) === null || _b === void 0 ? void 0 : _b.table;
                // `method` actually refers to the knex API method - Not exactly "operation"
                // in the spec sense, but matches most of the time.
                const operation = query === null || query === void 0 ? void 0 : query.method;
                const name = ((_c = config === null || config === void 0 ? void 0 : config.connection) === null || _c === void 0 ? void 0 : _c.filename) || ((_d = config === null || config === void 0 ? void 0 : config.connection) === null || _d === void 0 ? void 0 : _d.database);
                const maxLen = instrumentation._config.maxQueryLength;
                const attributes = {
                    'knex.version': moduleVersion,
                    [semantic_conventions_1.SemanticAttributes.DB_SYSTEM]: utils.mapSystem(config.client),
                    [semantic_conventions_1.SemanticAttributes.DB_SQL_TABLE]: table,
                    [semantic_conventions_1.SemanticAttributes.DB_OPERATION]: operation,
                    [semantic_conventions_1.SemanticAttributes.DB_USER]: (_e = config === null || config === void 0 ? void 0 : config.connection) === null || _e === void 0 ? void 0 : _e.user,
                    [semantic_conventions_1.SemanticAttributes.DB_NAME]: name,
                    [semantic_conventions_1.SemanticAttributes.NET_PEER_NAME]: (_f = config === null || config === void 0 ? void 0 : config.connection) === null || _f === void 0 ? void 0 : _f.host,
                    [semantic_conventions_1.SemanticAttributes.NET_PEER_PORT]: (_g = config === null || config === void 0 ? void 0 : config.connection) === null || _g === void 0 ? void 0 : _g.port,
                    [semantic_conventions_1.SemanticAttributes.NET_TRANSPORT]: ((_h = config === null || config === void 0 ? void 0 : config.connection) === null || _h === void 0 ? void 0 : _h.filename) === ':memory:' ? 'inproc' : undefined,
                };
                if (maxLen !== 0) {
                    attributes[semantic_conventions_1.SemanticAttributes.DB_STATEMENT] = utils.limitLength(query === null || query === void 0 ? void 0 : query.sql, maxLen);
                }
                const parent = this.builder[contextSymbol];
                const span = instrumentation.tracer.startSpan(utils.getName(name, operation, table), {
                    attributes,
                }, parent);
                const spanContext = api.trace.setSpan(api.context.active(), span);
                return api.context
                    .with(spanContext, original, this, ...arguments)
                    .then((result) => {
                    span.end();
                    return result;
                })
                    .catch((err) => {
                    // knex adds full query with all the binding values to the message,
                    // we want to undo that without changing the original error
                    const formatter = utils.getFormatter(this);
                    const fullQuery = formatter(query.sql, query.bindings || []);
                    const message = err.message.replace(fullQuery + ' - ', '');
                    const clonedError = utils.cloneErrorWithNewMessage(err, message);
                    span.recordException(clonedError);
                    span.setStatus({ code: api.SpanStatusCode.ERROR, message });
                    span.end();
                    throw err;
                });
            };
        };
    }
    storeContext(original) {
        return function wrapped_logging_method() {
            const builder = original.apply(this, arguments);
            // Builder is a custom promise type and when awaited it fails to propagate context.
            // We store the parent context at the moment of initiating the builder
            // otherwise we'd have nothing to attach the span as a child for in `query`.
            Object.defineProperty(builder, contextSymbol, {
                value: api.context.active(),
            });
            return builder;
        };
    }
    ensureWrapped(moduleVersion, obj, methodName, wrapper) {
        api.diag.debug(`Applying ${methodName} patch for ${constants.MODULE_NAME}@${moduleVersion}`);
        if ((0, instrumentation_1.isWrapped)(obj[methodName])) {
            this._unwrap(obj, methodName);
        }
        this._wrap(obj, methodName, wrapper);
    }
}
exports.KnexInstrumentation = KnexInstrumentation;
//# sourceMappingURL=instrumentation.js.map