"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "patientPaymentRoute", {
    enumerable: true,
    get: function() {
        return patientPaymentRoute;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _constants = require("@tamanu/constants");
const _zod = require("zod");
const _errors = require("@tamanu/shared/errors");
const _invoice = require("@tamanu/shared/utils/invoice");
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _decimal = /*#__PURE__*/ _interop_require_default(require("decimal.js"));
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const createPatientPaymentSchema = _zod.z.object({
    date: _zod.z.string().date(),
    amount: _zod.z.coerce.number().transform((amount)=>(0, _invoice.round)(amount, 2)),
    receiptNumber: _zod.z.string().regex(/^[A-Za-z0-9]+$/),
    methodId: _zod.z.string()
}).strip();
const updatePatientPaymentSchema = _zod.z.object({
    date: _zod.z.string().date(),
    amount: _zod.z.coerce.number().transform((amount)=>(0, _invoice.round)(amount, 2)),
    receiptNumber: _zod.z.string().regex(/^[A-Za-z0-9]+$/),
    methodId: _zod.z.string()
}).strip();
async function getInvoiceWithDetails(req, invoiceId) {
    return await req.models.Invoice.findByPk(invoiceId, {
        include: req.models.Invoice.getFullReferenceAssociations(req.models)
    });
}
const handleCreatePatientPayment = (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'InvoicePayment');
    const invoiceId = req.params.invoiceId;
    const invoice = await getInvoiceWithDetails(req, invoiceId);
    if (!invoice) throw new _errors.NotFoundError('Invoice not found');
    if (invoice.status !== _constants.INVOICE_STATUSES.FINALISED) throw new _errors.ForbiddenError('Invoice is not finalised');
    const { data, error } = await createPatientPaymentSchema.safeParseAsync(req.body);
    if (error) throw new _errors.ValidationError(error.message);
    const { patientTotal, patientPaymentRemainingBalance } = (0, _invoice.getInvoiceSummary)(invoice);
    if (data.amount > (0, _invoice.round)(patientPaymentRemainingBalance, 2)) throw new _errors.ForbiddenError('Amount of payment is higher than the owing total');
    const transaction = await req.db.transaction();
    try {
        const payment = await req.models.InvoicePayment.create({
            invoiceId,
            date: data.date,
            receiptNumber: data.receiptNumber,
            amount: data.amount
        }, {
            returning: true,
            transaction
        });
        await req.models.InvoicePatientPayment.create({
            invoicePaymentId: payment.id,
            methodId: data.methodId
        }, {
            transaction
        });
        //Update the overall patient payment status to invoice
        await req.models.Invoice.update({
            patientPaymentStatus: (0, _invoice.getInvoicePatientPaymentStatus)(new _decimal.default(patientTotal).minus(patientPaymentRemainingBalance).add(data.amount).toNumber(), patientTotal)
        }, {
            where: {
                id: invoiceId
            }
        }, {
            transaction
        });
        await transaction.commit();
        res.json(payment);
    } catch (error) {
        await transaction.rollback();
        throw error;
    }
});
const handleUpdatePatientPayment = (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'InvoicePayment');
    const invoiceId = req.params.invoiceId;
    const paymentId = req.params.paymentId;
    const invoice = await getInvoiceWithDetails(req, invoiceId);
    const payment = await req.models.InvoicePayment.findByPk(paymentId);
    if (!invoice) throw new _errors.NotFoundError('Invoice not found');
    if (invoice.status !== _constants.INVOICE_STATUSES.FINALISED) throw new _errors.ForbiddenError('Invoice is not finalised');
    if (!payment) throw new _errors.NotFoundError('Payment not found');
    const { data, error } = await updatePatientPaymentSchema.safeParseAsync(req.body);
    if (error) throw new _errors.ValidationError(error.message);
    const { patientTotal, patientPaymentRemainingBalance } = (0, _invoice.getInvoiceSummary)(invoice);
    if (data.amount > (0, _invoice.round)(new _decimal.default(patientPaymentRemainingBalance).add(payment.amount).toNumber(), 2)) throw new _errors.ForbiddenError('Amount of payment is higher than the invoice total');
    const transaction = await req.db.transaction();
    try {
        await req.models.InvoicePayment.update({
            date: data.date,
            receiptNumber: data.receiptNumber,
            amount: data.amount
        }, {
            where: {
                id: paymentId
            }
        }, {
            transaction,
            returning: true
        });
        await req.models.InvoicePatientPayment.update({
            methodId: data.methodId
        }, {
            where: {
                invoicePaymentId: paymentId
            }
        }, {
            transaction
        });
        //Update the overall patient payment status to invoice
        await req.models.Invoice.update({
            patientPaymentStatus: (0, _invoice.getInvoicePatientPaymentStatus)(new _decimal.default(patientTotal).minus(patientPaymentRemainingBalance).minus(payment.amount).add(data.amount).toNumber(), patientTotal)
        }, {
            where: {
                id: invoiceId
            }
        }, {
            transaction
        });
        await transaction.commit();
        res.json(payment);
    } catch (error) {
        await transaction.rollback();
        throw error;
    }
});
const handleGetPatientPayments = (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'InvoicePayment');
    const invoiceId = req.params.invoiceId;
    const patientPayments = await req.models.InvoicePatientPayment.findAll({
        include: [
            {
                model: req.models.InvoicePayment,
                as: 'detail',
                where: {
                    invoiceId
                }
            },
            {
                model: req.models.ReferenceData,
                as: 'method'
            }
        ]
    }).then((payments)=>payments.map((payment)=>({
                id: payment.invoicePaymentId,
                date: payment.detail.date,
                amount: payment.detail.amount,
                receiptNumber: payment.detail.receiptNumber,
                methodId: payment.methodId,
                methodName: payment.method.name
            })));
    res.json({
        count: patientPayments.length,
        data: patientPayments
    });
});
const patientPaymentRoute = _express.default.Router();
patientPaymentRoute.post('/:invoiceId/patientPayments', handleCreatePatientPayment);
patientPaymentRoute.put('/:invoiceId/patientPayments/:paymentId', handleUpdatePatientPayment);
patientPaymentRoute.get('/:invoiceId/patientPayments', handleGetPatientPayments);

//# sourceMappingURL=patientPayment.js.map