"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    fromSearchId: function() {
        return fromSearchId;
    },
    getDefaultOperator: function() {
        return getDefaultOperator;
    },
    getParamAndModifier: function() {
        return getParamAndModifier;
    },
    getQueryObject: function() {
        return getQueryObject;
    },
    isValidHl7Date: function() {
        return isValidHl7Date;
    },
    parseHL7Reference: function() {
        return parseHL7Reference;
    },
    toSearchId: function() {
        return toSearchId;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _encodings = require("@tamanu/shared/utils/encodings");
const _errors = require("@tamanu/shared/errors");
const _dateTime = require("@tamanu/shared/utils/dateTime");
const _datefns = require("date-fns");
const _datefnstz = require("date-fns-tz");
function toSearchId({ after, ...params }) {
    const result = {
        ...params
    };
    if (after) {
        result.after = {
            createdAt: after.createdAt.toISOString(),
            id: after.id
        };
    }
    return (0, _encodings.jsonToBase64)(result);
}
function fromSearchId(cursor) {
    // leave it to parseQuery to validate params
    return (0, _encodings.jsonFromBase64)(cursor);
}
function getParamAndModifier(fullParam) {
    return fullParam.split(':', 2);
}
function getDefaultOperator(type) {
    if (type === _constants.FHIR_SEARCH_PARAMETERS.STRING) {
        return _sequelize.Op.startsWith;
    }
    if (type === _constants.FHIR_SEARCH_PARAMETERS.DATE) {
        return _sequelize.Op.between;
    }
    return _sequelize.Op.eq;
}
function getQueryObject(columnName, value, operator, modifier, parameterType) {
    // String searches should be case insensitive unless the modifier is "exact"
    if (parameterType === _constants.FHIR_SEARCH_PARAMETERS.STRING && modifier !== 'exact') {
        // Perform case insensitive search by using SQL function UPPER
        // and modifying the string to be uppercase.
        return _sequelize.Sequelize.where(_sequelize.Sequelize.fn('upper', _sequelize.Sequelize.col(columnName)), {
            [operator]: value.toUpperCase()
        });
    }
    // Dates with eq modifier or no modifier should be looked up as a range
    if (parameterType === _constants.FHIR_SEARCH_PARAMETERS.DATE && [
        'eq',
        undefined
    ].includes(modifier)) {
        const [startDate, endDate] = getHL7DateRange(value);
        if ([
            'date_of_birth',
            'date_of_death'
        ].includes(columnName)) {
            return {
                [operator]: [
                    (0, _dateTime.toDateString)(startDate),
                    (0, _dateTime.toDateString)(endDate)
                ]
            };
        }
        return {
            [operator]: [
                startDate,
                endDate
            ]
        };
    }
    return {
        [operator]: value
    };
}
function getHL7DateRange(dateString) {
    if (!isValidHl7Date(dateString)) return null;
    const [startOf, endOf] = getStartEndOfFns(dateString);
    return [
        (0, _datefnstz.zonedTimeToUtc)(startOf((0, _datefns.parseISO)(dateString)), 'UTC'),
        (0, _datefnstz.zonedTimeToUtc)(endOf((0, _datefns.parseISO)(dateString)), 'UTC')
    ];
}
function isValidHl7Date(dateString) {
    // Only these formats should be valid for a date in HL7 FHIR:
    // https://www.hl7.org/fhir/datatypes.html#date
    return [
        'yyyy',
        'yyyy-MM',
        'yyyy-MM-dd'
    ].some((format)=>(0, _datefns.isMatch)(dateString, format));
}
// Returns the smallest time unit used on the date string format.
// Only supports HL7 formats.
function getStartEndOfFns(dateString) {
    switch(dateString.length){
        case 4:
            return [
                _datefns.startOfYear,
                _datefns.endOfYear
            ];
        case 7:
            return [
                _datefns.startOfMonth,
                _datefns.endOfMonth
            ];
        case 10:
            return [
                _datefns.startOfDay,
                _datefns.endOfDay
            ];
        default:
            throw new _errors.InvalidParameterError(`Invalid date/time format: ${dateString}`);
    }
}
function parseHL7Reference(reference) {
    const params = reference.split('/');
    return params[params.length - 1];
}

//# sourceMappingURL=search.js.map