"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "VRSActionHandler", {
    enumerable: true,
    get: function() {
        return VRSActionHandler;
    }
});
const _util = /*#__PURE__*/ _interop_require_default(require("util"));
const _logging = require("@tamanu/shared/services/logging");
const _errors = require("@tamanu/shared/errors");
const _schema = /*#__PURE__*/ _interop_require_wildcard(require("./schema"));
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
let VRSActionHandler = class VRSActionHandler {
    async retryPendingActions() {
        const actions = await this.remote.getAllPendingActions();
        if (actions.length === 0) {
            return; // quit early and don't log anything
        }
        _logging.log.info(`VRSActionHandler: Retrying ${actions.length} actions`);
        // retry one action at a time
        let successCount = 0;
        let skipCount = 0;
        let failCount = 0;
        for (const action of actions){
            try {
                const { CreatedDateTime, Operation, Id } = action;
                const isRecent = CreatedDateTime.getTime() + this.retryMinAgeMs > Date.now();
                if (isRecent) {
                    _logging.log.debug(`VRSActionHandler: Skipping recent action (action=${JSON.stringify(action)})`);
                    skipCount++;
                } else {
                    _logging.log.debug(`VRSActionHandler: Retrying action (action=${JSON.stringify(action)})`);
                    await this.applyAction({
                        created_datetime: CreatedDateTime,
                        operation: Operation,
                        fetch_id: Id
                    });
                    successCount++;
                }
            } catch (e) {
                _logging.log.error('VRSActionHandler: Recieved error while applying action', {
                    fetchId: action?.Id
                });
                _logging.log.error(e);
                failCount++;
            }
        }
        _logging.log.info(`VRSActionHandler: Finished (${actions.length} total, ${successCount} successful, ${skipCount} skipped, ${failCount} failed)`);
    }
    async applyAction(action) {
        const { sequelize, models } = this.store;
        const { Patient, PatientAdditionalData, PatientVRSData } = models;
        // validate action
        const { operation, fetch_id: fetchId } = await _schema.remoteRequest.patientCreated.validate(action, {
            stripUnknown: true
        });
        _logging.log.debug(`VRSActionHandler: applying action (operation=${operation}, fetch_id=${fetchId})`);
        // fetch patient
        const { patient, patientAdditionalData, patientVRSData } = await this.remote.getPatientByFetchId(fetchId);
        // persist
        if (operation === _schema.OPERATIONS.DELETE) {
            if (this.flagInsteadOfDeleting) {
                const { id: patientId } = await Patient.findOne({
                    where: {
                        displayId: patient.displayId
                    }
                });
                await PatientVRSData.upsert({
                    patientId,
                    isDeletedByRemote: true
                });
            } else {
                await Patient.update({
                    deletedAt: new Date()
                }, {
                    where: {
                        displayId: patient.displayId
                    }
                });
            }
        } else if ([
            _schema.OPERATIONS.INSERT,
            _schema.OPERATIONS.UPDATE
        ].includes(operation)) {
            await sequelize.transaction(async ()=>{
                // allow inserts and updates to resurrect deleted records - real deletion path
                const [{ id: upsertedPatientId }] = await Patient.upsert({
                    ...patient,
                    deletedAt: null
                }, {
                    returning: true,
                    paranoid: false
                });
                patientAdditionalData.patientId = upsertedPatientId;
                patientVRSData.patientId = upsertedPatientId;
                await PatientAdditionalData.upsert(patientAdditionalData);
                // allow inserts and updates to resurrect deleted records - data flag path
                await PatientVRSData.upsert({
                    ...patientVRSData,
                    isDeletedByRemote: false
                });
            });
        } else {
            throw new _errors.InvalidOperationError(`vrs: Operation not supported: ${operation}`);
        }
        // acknowledge request
        try {
            await this.remote.acknowledge(fetchId);
        } catch (e) {
            throw new _errors.RemoteCallFailedError(`vrs: Patient import succeded, but received an error while acknowledging: (displayId=${patient.displayId}, error=${_util.default.inspect(e)}`);
        }
    }
    constructor(store, remote, { flagInsteadOfDeleting, retryMinAgeMs }){
        _define_property(this, "store", null);
        _define_property(this, "remote", null);
        this.store = store;
        this.remote = remote;
        this.flagInsteadOfDeleting = flagInsteadOfDeleting;
        this.retryMinAgeMs = retryMinAgeMs;
    }
};

//# sourceMappingURL=VRSActionHandler.js.map