"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    ISO8061_WITH_TIMEZONE: function() {
        return ISO8061_WITH_TIMEZONE;
    },
    ISO9075_DATETIME_FORMAT: function() {
        return ISO9075_DATETIME_FORMAT;
    },
    ISO9075_DATE_FORMAT: function() {
        return ISO9075_DATE_FORMAT;
    },
    ageInMonths: function() {
        return ageInMonths;
    },
    ageInWeeks: function() {
        return ageInWeeks;
    },
    ageInYears: function() {
        return ageInYears;
    },
    compareDateStrings: function() {
        return compareDateStrings;
    },
    convertISO9075toRFC3339: function() {
        return convertISO9075toRFC3339;
    },
    differenceInMilliseconds: function() {
        return differenceInMilliseconds;
    },
    doAgeRangesHaveGaps: function() {
        return doAgeRangesHaveGaps;
    },
    doAgeRangesOverlap: function() {
        return doAgeRangesOverlap;
    },
    format: function() {
        return format;
    },
    formatLong: function() {
        return formatLong;
    },
    formatShort: function() {
        return formatShort;
    },
    formatShortest: function() {
        return formatShortest;
    },
    formatTime: function() {
        return formatTime;
    },
    formatTimeWithSeconds: function() {
        return formatTimeWithSeconds;
    },
    getCurrentDateString: function() {
        return getCurrentDateString;
    },
    getCurrentDateTimeString: function() {
        return getCurrentDateTimeString;
    },
    getCurrentISO8601DateString: function() {
        return getCurrentISO8601DateString;
    },
    getDateSubtractedFromToday: function() {
        return getDateSubtractedFromToday;
    },
    getDateTimeSubtractedFromNow: function() {
        return getDateTimeSubtractedFromNow;
    },
    isISOString: function() {
        return isISOString;
    },
    parseDate: function() {
        return parseDate;
    },
    toDateString: function() {
        return toDateString;
    },
    toDateTimeString: function() {
        return toDateTimeString;
    }
});
const _datefns = require("date-fns");
const _constants = require("@tamanu/constants");
const ISO9075_DATE_FORMAT = 'yyyy-MM-dd';
const ISO9075_DATETIME_FORMAT = 'yyyy-MM-dd HH:mm:ss';
const ISO8061_WITH_TIMEZONE = "yyyy-MM-dd'T'HH:mm:ssXXX";
const isISOString = (dateString)=>(0, _datefns.isMatch)(dateString, ISO9075_DATETIME_FORMAT) || (0, _datefns.isMatch)(dateString, ISO9075_DATE_FORMAT);
const parseDate = (date)=>{
    if (date === null || date === undefined) {
        return null;
    }
    let dateObj = date;
    if (isISOString(date)) {
        dateObj = (0, _datefns.parseISO)(date);
    } else if (typeof date === 'string') {
        // It seems that some JS implementations have problems parsing strings to dates.
        dateObj = new Date(date.replace(' ', 'T'));
    }
    if (!(0, _datefns.isValid)(dateObj)) {
        throw new Error('Not a valid date');
    }
    return dateObj;
};
function toDateTimeString(date) {
    if (date === null || date === undefined) {
        return null;
    }
    const dateObj = parseDate(date);
    return (0, _datefns.formatISO9075)(dateObj, {
        representation: 'complete'
    });
}
function toDateString(date) {
    if (date === null || date === undefined) {
        return null;
    }
    const dateObj = parseDate(date);
    return (0, _datefns.formatISO9075)(dateObj, {
        representation: 'date'
    });
}
function getCurrentDateTimeString() {
    return (0, _datefns.formatISO9075)(new Date());
}
function getDateTimeSubtractedFromNow(daysToSubtract) {
    return toDateTimeString((0, _datefns.sub)(new Date(), {
        days: daysToSubtract
    }));
}
function getDateSubtractedFromToday(daysToSubtract) {
    return toDateTimeString((0, _datefns.sub)((0, _datefns.startOfDay)(new Date()), {
        days: daysToSubtract
    }));
}
function getCurrentDateString() {
    return (0, _datefns.formatISO9075)(new Date(), {
        representation: 'date'
    });
}
function getCurrentISO8601DateString() {
    return new Date().toISOString();
}
function convertISO9075toRFC3339(dateString) {
    // parseISO does not support null values
    const parsedDate = dateString === null ? new Date() : (0, _datefns.parseISO)(dateString);
    return (0, _datefns.format)(parsedDate, "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
}
function ageInWeeks(dob) {
    return (0, _datefns.differenceInWeeks)(new Date(), (0, _datefns.parseISO)(dob));
}
function ageInMonths(dob) {
    return (0, _datefns.differenceInMonths)(new Date(), (0, _datefns.parseISO)(dob));
}
function ageInYears(dob) {
    return (0, _datefns.differenceInYears)(new Date(), (0, _datefns.parseISO)(dob));
}
function compareDateStrings(key = 'desc') {
    return (a, b)=>{
        if (key.toLowerCase() === 'asc') return (0, _datefns.parseISO)(a.date) - (0, _datefns.parseISO)(b.date);
        if (key.toLowerCase() === 'desc') return (0, _datefns.parseISO)(b.date) - (0, _datefns.parseISO)(a.date);
        return 0;
    };
}
function getAgeRangeInMinutes({ ageMin = -Infinity, ageMax = Infinity, ageUnit }) {
    const timeUnit = _constants.TIME_UNIT_OPTIONS.find((option)=>option.unit === ageUnit);
    const conversionValue = timeUnit.minutes;
    return {
        ageMin: ageMin * conversionValue,
        ageMax: ageMax * conversionValue,
        previousAgeUnit: ageUnit
    };
}
function doAgeRangesHaveGaps(rangesArray) {
    const conversions = {
        weeks: {
            months: (a, b)=>{
                const weeks = a.ageMax / 60 / 24 / 7;
                const months = b.ageMin / 60 / 24 / 30;
                return weeks / 4 !== months;
            },
            years: (a, b)=>{
                const weeks = a.ageMax / 60 / 24 / 7;
                const years = b.ageMin / 60 / 24 / 365;
                return weeks / 52 !== years;
            }
        },
        months: {
            years: (a, b)=>{
                const months = a.ageMax / 60 / 24 / 30;
                const years = b.ageMin / 60 / 24 / 365;
                return months / 12 !== years;
            }
        }
    };
    // Get all values into same time unit and sort by ageMin low to high
    const normalized = rangesArray.map(getAgeRangeInMinutes);
    normalized.sort((a, b)=>a.ageMin - b.ageMin);
    return normalized.some((rangeA, i)=>{
        const rangeB = normalized[i + 1];
        // This means we reached the last item, nothing more to compare
        if (!rangeB) return false;
        if (rangeA.previousAgeUnit !== rangeB.previousAgeUnit) {
            // No conversion means that minute comparison is good
            const conversion = conversions[rangeA.previousAgeUnit]?.[rangeB.previousAgeUnit];
            if (conversion) {
                return conversion(rangeA, rangeB);
            }
        }
        // These have to forcefully match, otherwise a gap exists
        return rangeA.ageMax !== rangeB.ageMin;
    });
}
function doAgeRangesOverlap(rangesArray) {
    return rangesArray.some((rangeA, aIndex)=>{
        return rangesArray.some((rangeB, bIndex)=>{
            // Only compare once between two ranges
            if (aIndex >= bIndex) return false;
            // Get both values into same time unit
            const aInMinutes = getAgeRangeInMinutes(rangeA);
            const bInMinutes = getAgeRangeInMinutes(rangeB);
            // Figure out the lowest min range
            const lowestMin = aInMinutes.ageMin < bInMinutes.ageMin ? aInMinutes : bInMinutes;
            const highestMin = aInMinutes.ageMin < bInMinutes.ageMin ? bInMinutes : aInMinutes;
            const lowestAgeMax = lowestMin.ageMax;
            const highestAgeMin = highestMin.ageMin;
            // Min inclusive - max exclusive: only overlaps if its less than
            return highestAgeMin < lowestAgeMax;
        });
    });
}
const format = (date, f)=>{
    if (date === null || date === undefined) {
        return null;
    }
    const dateObj = parseDate(date);
    return (0, _datefns.format)(dateObj, f);
};
const differenceInMilliseconds = (a, b)=>(0, _datefns.differenceInMilliseconds)(new Date(a), new Date(b));
const locale = globalThis.navigator?.language ?? 'default';
const intlFormatDate = (date, formatOptions, fallback = 'Unknown')=>{
    if (!date) return fallback;
    return (0, _datefns.parseISO)(date).toLocaleString(locale, formatOptions);
};
const formatShortest = (date)=>intlFormatDate(date, {
        month: '2-digit',
        day: '2-digit',
        year: '2-digit'
    }, '--/--'); // 12/04/20
const formatShort = (date)=>intlFormatDate(date, {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    }, '--/--/----'); // 12/04/2020
const formatTime = (date)=>intlFormatDate(date, {
        timeStyle: 'short',
        hour12: true
    }, '__:__'); // 12:30 am
const formatTimeWithSeconds = (date)=>intlFormatDate(date, {
        timeStyle: 'medium',
        hour12: true
    }, '__:__:__'); // 12:30:00 am
const formatLong = (date)=>intlFormatDate(date, {
        timeStyle: 'short',
        dateStyle: 'full',
        hour12: true
    }, 'Date information not available'); // "Thursday, 14 July 2022, 03:44 pm"

//# sourceMappingURL=dateTime.js.map