"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "user", {
    enumerable: true,
    get: function() {
        return user;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _errors = require("@tamanu/shared/errors");
const _middleware = require("@tamanu/shared/permissions/middleware");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _query = require("../../utils/query");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const user = _express.default.Router();
user.get('/me', (0, _expressasynchandler.default)(async (req, res)=>{
    if (!req.user) {
        throw new _errors.BadAuthenticationError('Invalid token (LLh7)');
    }
    req.checkPermission('read', req.user);
    res.send(req.user);
}));
user.get('/permissions', (0, _expressasynchandler.default)(_middleware.getPermissions));
user.get('/current-facility', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('list', 'User');
    const userFacilities = await req.models.UserFacility.findAll({
        where: {
            facilityId: _config.default.serverFacilityId
        },
        include: [
            {
                model: req.models.User,
                as: 'user'
            }
        ]
    });
    const users = userFacilities.map((userFacility)=>userFacility.get({
            plain: true
        }).user);
    res.send(users);
}));
user.get('/recently-viewed-patients', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { Patient }, user: currentUser, query } = req;
    req.checkPermission('read', currentUser);
    req.checkPermission('list', 'Patient');
    const filters = [
        (0, _query.makeFilter)(query.encounterType, 'encounters.encounter_type = :encounterType', ()=>({
                encounterType: query.encounterType
            })),
        (0, _query.makeDeletedAtIsNullFilter)('encounters'),
        (0, _query.makeFilter)(true, `user_recently_viewed_patients.user_id = :userId`, ()=>({
                userId: currentUser.id
            }))
    ];
    const { whereClauses, filterReplacements } = (0, _query.getWhereClausesAndReplacementsFromFilters)(filters);
    const recentlyViewedPatients = await req.db.query(`
      SELECT
        patients.id,
        patients.display_id,
        patients.first_name,
        patients.last_name,
        patients.sex,
        patients.date_of_birth,
        encounters.id AS encounter_id,
        encounters.encounter_type,
        user_recently_viewed_patients.updated_at AS last_accessed_on
      FROM user_recently_viewed_patients
        LEFT JOIN patients
          ON (patients.id = user_recently_viewed_patients.patient_id)
        LEFT JOIN (
            SELECT *, ROW_NUMBER() OVER (PARTITION BY patient_id ORDER BY start_date DESC, id DESC) AS row_num
            FROM encounters
            WHERE end_date IS NULL
            AND deleted_at IS NULL
            ) encounters
            ON (patients.id = encounters.patient_id AND encounters.row_num = 1)
        ${whereClauses && `WHERE ${whereClauses}`}
        ORDER BY last_accessed_on DESC
        LIMIT 12
      `, {
        model: Patient,
        type: _sequelize.QueryTypes.SELECT,
        mapToModel: true,
        replacements: filterReplacements
    });
    res.send({
        data: recentlyViewedPatients,
        count: recentlyViewedPatients.length
    });
}));
user.post('/recently-viewed-patients/:patientId', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { UserRecentlyViewedPatient }, user: currentUser, params } = req;
    const { patientId } = params;
    req.checkPermission('read', 'Patient');
    const [createdRelation] = await UserRecentlyViewedPatient.create({
        userId: currentUser.id,
        patientId
    });
    res.send(createdRelation);
}));
user.get('/userPreferences', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { UserPreference }, user: currentUser } = req;
    req.checkPermission('read', currentUser);
    const userPreferences = await UserPreference.findOne({
        where: {
            userId: currentUser.id
        }
    });
    // Return {} as default if no user preferences exist
    res.send(userPreferences || {});
}));
user.post('/userPreferences', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { UserPreference }, user: currentUser, body } = req;
    req.checkPermission('write', currentUser);
    const { selectedGraphedVitalsOnFilter } = body;
    const [userPreferences] = await UserPreference.upsert({
        selectedGraphedVitalsOnFilter,
        userId: currentUser.id,
        deletedAt: null
    });
    res.send(userPreferences);
}));
user.get('/:id', (0, _crudHelpers.simpleGet)('User'));
const globalUserRequests = (0, _crudHelpers.permissionCheckingRouter)('list', 'User');
globalUserRequests.get('/$', (0, _crudHelpers.paginatedGetList)('User'));
user.use(globalUserRequests);

//# sourceMappingURL=user.js.map