"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Triage", {
    enumerable: true,
    get: function() {
        return Triage;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("./buildEncounterLinkedSyncFilter");
const _dateTimeTypes = require("./dateTimeTypes");
const _buildPatientLinkedLookupFilter = require("./buildPatientLinkedLookupFilter");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let Triage = class Triage extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            arrivalTime: (0, _dateTimeTypes.dateTimeType)('arrivalTime'),
            triageTime: (0, _dateTimeTypes.dateTimeType)('triageTime'),
            closedTime: (0, _dateTimeTypes.dateTimeType)('closedTime'),
            score: _sequelize.Sequelize.TEXT
        }, {
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            ...options
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.belongsTo(models.User, {
            as: 'Practitioner',
            foreignKey: 'practitionerId'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'chiefComplaintId'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'secondaryComplaintId'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'arrivalModeId',
            as: 'arrivalMode'
        });
        this.hasMany(models.Note, {
            foreignKey: 'recordId',
            as: 'notes',
            constraints: false,
            scope: {
                recordType: this.name
            }
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildPatientLinkedLookupFilter.buildEncounterPatientIdSelect)(this),
            joins: (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
                this.tableName,
                'encounters'
            ])
        };
    }
    static async create(data) {
        const { Department, Encounter, ReferenceData } = this.sequelize.models;
        const existingEncounter = await Encounter.findOne({
            where: {
                endDate: {
                    [_sequelize.Op.is]: null
                },
                patientId: data.patientId
            }
        });
        if (existingEncounter) {
            throw new _errors.InvalidOperationError("Can't triage a patient that has an existing encounter");
        }
        const reasons = await Promise.all([
            data.chiefComplaintId,
            data.secondaryComplaintId
        ].map((x)=>ReferenceData.findByPk(x)));
        // TODO: to handle translations for triage reason for encounter
        const reasonsText = reasons.filter((x)=>x).map((x)=>x.name).join(' and ');
        const reasonForEncounter = `Presented at emergency department with ${reasonsText}`;
        let { departmentId } = data;
        if (!departmentId) {
            const department = await Department.findOne({
                where: {
                    name: 'Emergency',
                    facilityId: _config.default.serverFacilityId
                }
            });
            if (!department) {
                throw new Error('Cannot find Emergency department for current facility');
            }
            // eslint-disable-next-line require-atomic-updates
            departmentId = department.id;
        }
        return this.sequelize.transaction(async ()=>{
            const encounter = await Encounter.create({
                encounterType: _constants.ENCOUNTER_TYPES.TRIAGE,
                startDate: data.triageTime,
                reasonForEncounter,
                patientId: data.patientId,
                departmentId,
                locationId: data.locationId,
                examinerId: data.practitionerId,
                actorId: data.actorId
            });
            return super.create({
                ...data,
                encounterId: encounter.id
            });
        });
    }
};

//# sourceMappingURL=Triage.js.map