"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "VdsNcDocument", {
    enumerable: true,
    get: function() {
        return VdsNcDocument;
    }
});
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _models = require("@tamanu/shared/models");
const _utils = require("@tamanu/shared/utils");
const _jsoncanonicalize = require("json-canonicalize");
const _Crypto = require("./Crypto");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
let VdsNcDocument = class VdsNcDocument {
    /**
   * Returns the "message to sign" part of the document.
   *
   * @internal
   * @returns {Object}
   */ getMessageToSign() {
        if (!this.signer) throw new Error('Must have a signer.');
        const msg = this.messageData;
        switch(this.type){
            case 'icao.test':
                msg.utci = this.uniqueProofId;
                break;
            case 'icao.vacc':
                msg.uvci = this.uniqueProofId;
                break;
            default:
                throw new Error('Unreachable');
        }
        return {
            hdr: {
                t: this.type,
                v: 1,
                is: this.signer.countryCode
            },
            msg
        };
    }
    /**
   * Signs a document.
   *
   * If the document is already signed, this will silently do nothing, and return
   * as normal.
   *
   * @returns {Promise<Document>} This object, signed.
   * @throws {Error} if there's no active signer.
   */ async sign() {
        if (this.isSigned) {
            _logging.log.debug('Not signing VDS document: already signed');
            return this;
        }
        _logging.log.debug('Signing VDS document');
        const signer = await this.models.Signer.findActive();
        if (!signer) throw new Error('No active signer');
        this.signer = signer;
        const data = this.getMessageToSign();
        const { algorithm, signature } = await (0, _Crypto.issueVdsNcSignature)(data, {
            models: this.models
        });
        this.algorithm = algorithm;
        this.signature = signature;
        this.isSigned = true;
        return this;
    }
    /**
   * Returns the signed VDS-NC document as a string.
   *
   * This can then be encoded as a QR code.
   *
   * @returns {Promise<string>} Signed VDS-NC document.
   * @throws {Error} if it is not yet signed.
   */ async intoVDS() {
        if (!this.isSigned) throw new Error('Cannot return an unsigned VDS-NC document.');
        _logging.log.debug('Encoding VDS-NC document');
        return (0, _jsoncanonicalize.canonicalize)({
            data: this.getMessageToSign(),
            sig: {
                alg: this.algorithm,
                sigvl: (0, _utils.base64UrlEncode)(this.signature),
                cer: (0, _utils.base64UrlEncode)((0, _utils.depem)(this.signer.certificate, 'CERTIFICATE'))
            }
        });
    }
    constructor(type, messageData, uniqueProofId){
        _define_property(this, "models", {
            Signer: _models.Signer
        });
        _define_property(this, "isSigned", false);
        _logging.log.debug(`Initialising VDS document type=${type} uvci=${uniqueProofId}`);
        this.type = type;
        this.messageData = messageData;
        this.uniqueProofId = uniqueProofId;
        if (!Object.values(_constants.ICAO_DOCUMENT_TYPES).some((typ)=>this.type === typ.JSON)) {
            throw new Error('A VDS-NC document must have a valid type.');
        }
    }
};

//# sourceMappingURL=Document.js.map