"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    formatDisplayPrice: function() {
        return formatDisplayPrice;
    },
    getInsurerDiscountAmountDisplayList: function() {
        return getInsurerDiscountAmountDisplayList;
    },
    getInsurerPaymentsWithRemainingBalanceDisplay: function() {
        return getInsurerPaymentsWithRemainingBalanceDisplay;
    },
    getInvoiceInsurerPaymentStatus: function() {
        return getInvoiceInsurerPaymentStatus;
    },
    getInvoiceItemDiscountPriceDisplay: function() {
        return getInvoiceItemDiscountPriceDisplay;
    },
    getInvoiceItemPriceDisplay: function() {
        return getInvoiceItemPriceDisplay;
    },
    getInvoicePatientPaymentStatus: function() {
        return getInvoicePatientPaymentStatus;
    },
    getInvoiceSummary: function() {
        return getInvoiceSummary;
    },
    getInvoiceSummaryDisplay: function() {
        return getInvoiceSummaryDisplay;
    },
    getPatientPaymentsWithRemainingBalanceDisplay: function() {
        return getPatientPaymentsWithRemainingBalanceDisplay;
    },
    getSpecificInsurerPaymentRemainingBalance: function() {
        return getSpecificInsurerPaymentRemainingBalance;
    },
    isInvoiceEditable: function() {
        return isInvoiceEditable;
    },
    round: function() {
        return round;
    }
});
const _constants = require("@tamanu/constants");
const _decimal = /*#__PURE__*/ _interop_require_default(require("decimal.js"));
const _lodash = require("lodash");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const round = (value, decimals = 2)=>{
    return new _decimal.default(value).toNearest(new _decimal.default(10).pow(-decimals)).toNumber();
};
const isInvoiceEditable = (invoice)=>invoice.status === _constants.INVOICE_STATUSES.IN_PROGRESS;
const formatDisplayPrice = (value)=>isNaN(parseFloat(value)) ? undefined : round(value, 2).toFixed(2);
/**
 * get a price after applying a discount
 * @param {number} price
 * @param {number} discount
 * @returns {number}
 */ const discountedPrice = (price, discount)=>{
    return new _decimal.default(price).minus(discountAmount(price, discount)).toNumber();
};
/**
 * get a discounted price
 * @param {number} price
 * @param {number} discount
 */ const discountAmount = (price, discount)=>{
    return new _decimal.default(price).times(discount).toNumber();
};
/**
 * Get the price of an invoice item
 * @param {InvoiceItem} invoiceItem
 */ const getInvoiceItemTotalPrice = (invoiceItem)=>{
    return new _decimal.default(invoiceItem?.productPrice || 0).times(invoiceItem?.quantity || 1).toNumber();
};
/**
 * Get the price of an invoice item after applying the discount
 * @param {InvoiceItem} invoiceItem
 */ const getInvoiceItemTotalDiscountedPrice = (invoiceItem)=>{
    return discountedPrice(getInvoiceItemTotalPrice(invoiceItem), invoiceItem?.discount?.percentage || 0);
};
/**
 * Get the discount amount of an invoice insurer
 * @param {InvoiceInsurer} insurer
 * @param {number} total
 */ const getInvoiceInsurerDiscountAmount = (insurer, total)=>{
    return discountAmount(total || 0, insurer?.percentage || 0);
};
/**
 * Get the discount amount of an invoice discount
 * @param {InvoiceDiscount} discount
 * @param {number} total
 */ const getInvoiceDiscountDiscountAmount = (discount, total)=>{
    return discountAmount(total || 0, discount?.percentage || 0);
};
const getInvoiceSummary = (invoice)=>{
    invoice = JSON.parse(JSON.stringify(invoice)); // deep clone to convert sequelize entity to plain objects
    const discountableItemsSubtotal = invoice.items.filter((item)=>item?.productDiscountable).reduce((sum, item)=>sum.plus(getInvoiceItemTotalDiscountedPrice(item) || 0), new _decimal.default(0)).toNumber();
    const nonDiscountableItemsSubtotal = invoice.items.filter((item)=>!item?.productDiscountable).reduce((sum, item)=>sum.plus(getInvoiceItemTotalDiscountedPrice(item) || 0), new _decimal.default(0)).toNumber();
    const itemsSubtotal = new _decimal.default(discountableItemsSubtotal).add(nonDiscountableItemsSubtotal).toNumber();
    const insurersDiscountPercentage = invoice.insurers.reduce((sum, insurer)=>sum.plus(insurer?.percentage || 0), new _decimal.default(0)).toNumber();
    const insurerDiscountTotal = new _decimal.default(itemsSubtotal).times(insurersDiscountPercentage).toNumber();
    const patientSubtotal = new _decimal.default(itemsSubtotal).minus(insurerDiscountTotal).toNumber();
    const patientDiscountableSubtotal = discountedPrice(discountableItemsSubtotal, insurersDiscountPercentage);
    const discountTotal = getInvoiceDiscountDiscountAmount(invoice.discount, patientDiscountableSubtotal);
    const patientTotal = new _decimal.default(patientSubtotal).minus(discountTotal).toNumber();
    //Calculate payments as well
    const patientPaymentsTotal = invoice.payments.filter((payment)=>payment?.patientPayment?.id).reduce((sum, payment)=>sum.plus(payment.amount), new _decimal.default(0)).toNumber();
    const insurerPaymentsTotal = invoice.payments.filter((payment)=>payment?.insurerPayment?.id).reduce((sum, payment)=>sum.plus(payment.amount), new _decimal.default(0)).toNumber();
    const paymentsTotal = invoice.payments.reduce((sum, payment)=>sum.plus(payment.amount), new _decimal.default(0)).toNumber();
    const patientPaymentRemainingBalance = new _decimal.default(patientTotal).minus(patientPaymentsTotal).toNumber();
    const insurerPaymentRemainingBalance = new _decimal.default(insurerDiscountTotal).minus(insurerPaymentsTotal).toNumber();
    return {
        discountableItemsSubtotal,
        nonDiscountableItemsSubtotal,
        itemsSubtotal,
        insurerDiscountTotal,
        patientSubtotal,
        patientDiscountableSubtotal,
        discountTotal,
        patientTotal,
        patientPaymentsTotal,
        insurerPaymentsTotal,
        paymentsTotal,
        patientPaymentRemainingBalance,
        insurerPaymentRemainingBalance
    };
};
const getInvoiceSummaryDisplay = (invoice)=>{
    const discountableItems = invoice.items.filter((item)=>item.productDiscountable && !isNaN(parseFloat(item.productPrice)));
    const nonDiscountableItems = invoice.items.filter((item)=>!item.productDiscountable && !isNaN(parseFloat(item.productPrice)));
    const summary = getInvoiceSummary(invoice);
    return (0, _lodash.mapValues)(summary, (value, key)=>{
        if (!discountableItems.length && !nonDiscountableItems.length) return undefined;
        if (!discountableItems.length && key === 'discountableItemsSubtotal') return undefined;
        if (!nonDiscountableItems.length && key === 'nonDiscountableItemsSubtotal') return undefined;
        return formatDisplayPrice(value);
    });
};
const getSpecificInsurerPaymentRemainingBalance = (insurers, payments, insurerId, total)=>{
    const insurersDiscountPercentage = insurers.filter((insurer)=>insurer.insurerId === insurerId).reduce((sum, insurer)=>sum.plus(insurer?.percentage || 0), new _decimal.default(0)).toNumber();
    const insurerDiscountTotal = new _decimal.default(total).times(insurersDiscountPercentage).toNumber();
    const insurerPaymentsTotal = payments.filter((payment)=>payment?.insurerPayment?.id && payment.insurerPayment.insurerId === insurerId).reduce((sum, payment)=>sum.plus(payment.amount), new _decimal.default(0)).toNumber();
    return {
        insurerDiscountTotal,
        insurerPaymentsTotal,
        insurerPaymentRemainingBalance: new _decimal.default(insurerDiscountTotal).minus(insurerPaymentsTotal).toNumber()
    };
};
const getInvoiceItemPriceDisplay = (invoiceItem)=>{
    return formatDisplayPrice(isNaN(parseFloat(invoiceItem.productPrice)) ? undefined : getInvoiceItemTotalPrice(invoiceItem));
};
const getInvoiceItemDiscountPriceDisplay = (invoiceItem)=>{
    return formatDisplayPrice(isNaN(parseFloat(invoiceItem?.discount?.percentage)) ? undefined : getInvoiceItemTotalDiscountedPrice(invoiceItem));
};
const getInsurerDiscountAmountDisplayList = (insurers, total)=>{
    return insurers.map((insurer)=>getInvoiceInsurerDiscountAmount(insurer, total || 0)).map((value, index)=>formatDisplayPrice(isNaN(insurers[index]?.percentage) ? undefined : value));
};
const getInvoicePatientPaymentStatus = (paidAmount, owingAmount)=>{
    paidAmount = round(paidAmount, 2);
    owingAmount = round(owingAmount, 2);
    if (paidAmount < 0) throw new Error('Paid amount cannot be negative');
    if (paidAmount > owingAmount) throw new Error('Paid amount cannot be greater than owing amount');
    if (paidAmount === 0) return _constants.INVOICE_PATIENT_PAYMENT_STATUSES.UNPAID;
    if (paidAmount === owingAmount) return _constants.INVOICE_PATIENT_PAYMENT_STATUSES.PAID;
    return _constants.INVOICE_PATIENT_PAYMENT_STATUSES.PARTIAL;
};
const getInvoiceInsurerPaymentStatus = (paidAmount, owingAmount)=>{
    if (paidAmount == null) return _constants.INVOICE_INSURER_PAYMENT_STATUSES.UNPAID;
    paidAmount = round(paidAmount, 2);
    owingAmount = round(owingAmount, 2);
    if (paidAmount < 0) throw new Error('Paid amount cannot be negative');
    if (paidAmount > owingAmount) throw new Error('Paid amount cannot be greater than owing amount');
    if (paidAmount === 0) return _constants.INVOICE_INSURER_PAYMENT_STATUSES.REJECTED;
    if (paidAmount === owingAmount) return _constants.INVOICE_INSURER_PAYMENT_STATUSES.PAID;
    return _constants.INVOICE_INSURER_PAYMENT_STATUSES.PARTIAL;
};
const getPatientPaymentsWithRemainingBalanceDisplay = (invoice)=>{
    const patientPayments = invoice.payments.filter((payment)=>payment?.patientPayment?.id);
    let { patientTotal } = getInvoiceSummary(invoice);
    const patientPaymentsWithRemainingBalance = patientPayments?.map((payment)=>{
        patientTotal = new _decimal.default(patientTotal).minus(payment.amount).toNumber();
        return {
            ...payment,
            amount: formatDisplayPrice(payment.amount),
            remainingBalance: formatDisplayPrice(patientTotal)
        };
    });
    return patientPaymentsWithRemainingBalance;
};
const getInsurerPaymentsWithRemainingBalanceDisplay = (invoice)=>{
    const insurerPayments = invoice.payments.filter((payment)=>payment?.insurerPayment?.id);
    let { insurerDiscountTotal } = getInvoiceSummary(invoice);
    const insurerPaymentsWithRemainingBalance = insurerPayments?.map((payment)=>{
        insurerDiscountTotal = new _decimal.default(insurerDiscountTotal).minus(payment.amount).toNumber();
        return {
            ...payment,
            amount: formatDisplayPrice(payment.amount),
            remainingBalance: formatDisplayPrice(insurerDiscountTotal)
        };
    });
    return insurerPaymentsWithRemainingBalance;
};

//# sourceMappingURL=invoice.js.map