import PropTypes from 'prop-types';
import React from 'react';
import { Document, StyleSheet, View } from '@react-pdf/renderer';
import { getName, getTimeOfDeath, getDateOfDeath, getSex, getAddress, getNationality, getEthnicity, getClinician } from '../patientAccessors';
import { CertificateHeader, Col, Row, styles, SigningImage } from './Layout';
import { LetterheadSection } from './LetterheadSection';
import { Footer } from './printComponents/Footer';
import { MultiPageHeader } from './printComponents/MultiPageHeader';
import { renderDataItems } from './printComponents/renderDataItems';
import { P } from './Typography';
import { getDisplayDate } from './getDisplayDate';
import { DataSection } from './printComponents/DataSection';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
import { formatDistanceStrict, milliseconds } from 'date-fns';
const borderStyle = '1 solid black';
const tableLabelWidth = 200;
const tablePadding = 10;
const dataColPadding = 10;
const generalStyles = StyleSheet.create({
    container: {
        marginVertical: 8
    },
    tableContainer: {
        marginTop: 30
    },
    sectionContainer: {
        marginVertical: 7
    }
});
const TableContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: [
            generalStyles.container,
            generalStyles.tableContainer
        ],
        ...props
    });
const infoBoxStyles = StyleSheet.create({
    row: {
        flexDirection: 'row',
        border: borderStyle,
        marginBottom: -1
    },
    labelCol: {
        width: tableLabelWidth,
        padding: tablePadding
    },
    dataCol: {
        flex: 1,
        padding: dataColPadding,
        paddingBottom: 30
    },
    boldText: {
        fontFamily: 'Helvetica-Bold',
        fontSize: 12,
        fontWeight: 500
    },
    infoText: {
        fontFamily: 'Helvetica',
        fontSize: 12,
        fontWeight: 400
    },
    italicBoldText: {
        fontFamily: 'Helvetica-BoldOblique',
        fontSize: 12,
        fontWeight: 500
    },
    italicText: {
        fontFamily: 'Helvetica-Oblique',
        fontSize: 12,
        fontWeight: 500
    },
    underlinedText: {
        borderBottom: borderStyle,
        width: 175
    },
    marginTop: {
        marginTop: 50
    },
    mediumMarginTop: {
        marginTop: 30
    },
    smallMarginTop: {
        marginTop: 5
    }
});
const signStyles = StyleSheet.create({
    container: {
        paddingBottom: 20,
        marginVertical: 30
    },
    text: {
        fontFamily: 'Helvetica-Bold',
        fontSize: 12,
        fontWeight: 500,
        lineHeight: 1
    },
    line: {
        flex: 1,
        borderBottom: '1 solid black',
        height: 14,
        marginLeft: 5
    },
    row: {
        flexDirection: 'row',
        paddingTop: 20
    },
    leftCol: {
        flexDirection: 'row',
        width: 300,
        paddingRight: 20
    },
    rightCol: {
        flexDirection: 'row',
        flex: 1
    }
});
const InfoBoxRow = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.row,
        ...props
    });
const InfoBoxLabelCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.labelCol,
        ...props
    });
const UnderlinedText = ({ text, style, props })=>/*#__PURE__*/ React.createElement(View, {
        style: {
            ...infoBoxStyles.infoText,
            ...infoBoxStyles.underlinedText,
            ...style
        },
        ...props
    }, /*#__PURE__*/ React.createElement(Text, null, text));
const CauseField = ({ cause, label, helperText, ...props })=>{
    return /*#__PURE__*/ React.createElement(View, props, /*#__PURE__*/ React.createElement(Row, null, label && /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.infoText
    }, "(", label, ") "), /*#__PURE__*/ React.createElement(UnderlinedText, {
        text: cause.name
    }, " "), /*#__PURE__*/ React.createElement(UnderlinedText, {
        text: formatDistanceStrict(0, milliseconds({
            minutes: cause.timeAfterOnset
        })),
        style: {
            width: 85,
            marginLeft: 10
        }
    })), helperText && /*#__PURE__*/ React.createElement(Text, {
        style: [
            infoBoxStyles.infoText,
            infoBoxStyles.smallMarginTop
        ]
    }, helperText));
};
const InfoBoxDataCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.dataCol,
        ...props
    });
const AuthorisedAndSignSection = ()=>/*#__PURE__*/ React.createElement(View, {
        style: signStyles.container
    }, /*#__PURE__*/ React.createElement(View, {
        style: signStyles.row
    }, /*#__PURE__*/ React.createElement(P, {
        style: signStyles.text
    }, "Authorised by (print name):"), /*#__PURE__*/ React.createElement(View, {
        style: signStyles.line
    })), /*#__PURE__*/ React.createElement(View, {
        style: signStyles.row
    }, /*#__PURE__*/ React.createElement(View, {
        style: signStyles.leftCol
    }, /*#__PURE__*/ React.createElement(Text, {
        style: signStyles.text
    }, "Signed: "), /*#__PURE__*/ React.createElement(View, {
        style: signStyles.line
    })), /*#__PURE__*/ React.createElement(View, {
        style: signStyles.rightCol
    }, /*#__PURE__*/ React.createElement(Text, {
        style: signStyles.text
    }, "Date:"), /*#__PURE__*/ React.createElement(View, {
        style: signStyles.line
    }))));
const placeOfDeathAccessor = ({ facility })=>{
    return facility?.name;
};
const getCauseName = (cause)=>cause?.condition?.name;
const getCauseInfo = (cause)=>{
    const name = cause?.condition?.name;
    const timeAfterOnset = cause?.timeAfterOnset;
    return {
        name,
        timeAfterOnset
    };
};
const causeOfDeathAccessor = ({ causes })=>{
    return getCauseName(causes?.primary);
};
// Death certificate has a slightly different DOB format to other certificates so needs its own accessor
const getDOB = ({ dateOfBirth }, getLocalisation)=>dateOfBirth ? getDisplayDate(dateOfBirth, 'd MMM yyyy', getLocalisation) : 'Unknown';
const getDateAndTimeOfDeath = (patientData, getLocalisation)=>{
    return `${getDateOfDeath(patientData, getLocalisation)} ${getTimeOfDeath(patientData, getLocalisation)}`;
};
const PATIENT_DETAIL_FIELDS = {
    leftCol: [
        {
            key: 'firstName',
            label: 'First name'
        },
        {
            key: 'lastName',
            label: 'Last name'
        },
        {
            key: 'nationality',
            label: 'Nationality',
            accessor: getNationality
        },
        {
            key: 'address',
            label: 'Address',
            accessor: getAddress
        }
    ],
    rightCol: [
        {
            key: 'sex',
            label: 'Sex',
            accessor: getSex
        },
        {
            key: 'dateOfBirth',
            label: 'DOB',
            accessor: getDOB
        },
        {
            key: 'ethnicity',
            label: 'Ethnicity',
            accessor: getEthnicity
        }
    ]
};
const PATIENT_DEATH_DETAILS = {
    leftCol: [
        {
            key: 'deathDateAndTime',
            label: 'Date & time of death',
            accessor: getDateAndTimeOfDeath
        },
        {
            key: 'placeOfDeath',
            label: 'Place of death',
            accessor: placeOfDeathAccessor
        }
    ],
    rightCol: [
        {
            key: 'causeOfDeath',
            label: 'Cause of death',
            accessor: causeOfDeathAccessor
        },
        {
            key: 'clinician',
            label: 'Attending clinician',
            accessor: getClinician
        }
    ]
};
const SectionContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.sectionContainer,
        ...props
    });
const DeathCertificatePrintoutComponent = /*#__PURE__*/ React.memo(({ patientData, certificateData, getLocalisation })=>{
    const { logo, deathCertFooterImg } = certificateData;
    const { causes } = patientData;
    const causeOfDeath = getCauseInfo(causes?.primary);
    const antecedentCause1 = getCauseInfo(causes?.antecedent1);
    const antecedentCause2 = getCauseInfo(causes?.antecedent2);
    const antecedentCause3 = getCauseInfo(causes?.antecedent3);
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: {
            ...styles.page,
            paddingBottom: 25
        }
    }, /*#__PURE__*/ React.createElement(MultiPageHeader, {
        documentName: "Cause of death certificate",
        patientName: getName(patientData),
        patientId: patientData.displayId
    }), /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        getLocalisation: getLocalisation,
        logoSrc: logo,
        letterheadConfig: certificateData,
        certificateTitle: "Cause of death certificate"
    }), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(DataSection, {
        title: "Patient details",
        hideBottomRule: true
    }, /*#__PURE__*/ React.createElement(Col, null, renderDataItems(PATIENT_DETAIL_FIELDS.leftCol, patientData, getLocalisation, 12)), /*#__PURE__*/ React.createElement(Col, null, renderDataItems(PATIENT_DETAIL_FIELDS.rightCol, patientData, getLocalisation, 12))), /*#__PURE__*/ React.createElement(DataSection, {
        title: ""
    }, /*#__PURE__*/ React.createElement(Col, null, renderDataItems(PATIENT_DEATH_DETAILS.leftCol, patientData, getLocalisation, 12)), /*#__PURE__*/ React.createElement(Col, null, renderDataItems(PATIENT_DEATH_DETAILS.rightCol, patientData, getLocalisation, 12))))), /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(InfoBoxRow, null, /*#__PURE__*/ React.createElement(InfoBoxLabelCol, null, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.boldText
    }, "I ", '\n', "Disease or condition directly ", '\n', "leading to death*"), /*#__PURE__*/ React.createElement(Text, {
        style: [
            infoBoxStyles.italicBoldText,
            infoBoxStyles.marginTop
        ]
    }, "Antecedent Causes"), /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.infoText
    }, "Morbid conditions, if any,", '\n', "giving rise to the above cause,", '\n', "stating the underlying", '\n', "condition last")), /*#__PURE__*/ React.createElement(InfoBoxDataCol, null, /*#__PURE__*/ React.createElement(CauseField, {
        style: infoBoxStyles.mediumMarginTop,
        label: "a",
        helperText: "due to (or as a consequence of)",
        cause: causeOfDeath
    }), /*#__PURE__*/ React.createElement(CauseField, {
        style: infoBoxStyles.mediumMarginTop,
        label: "b",
        helperText: "due to (or as a consequence of)",
        cause: antecedentCause1
    }), /*#__PURE__*/ React.createElement(CauseField, {
        style: infoBoxStyles.mediumMarginTop,
        label: "c",
        helperText: "due to (or as a consequence of)",
        cause: antecedentCause2
    }), /*#__PURE__*/ React.createElement(CauseField, {
        style: infoBoxStyles.mediumMarginTop,
        label: "d",
        cause: antecedentCause3
    }))), /*#__PURE__*/ React.createElement(InfoBoxRow, null, /*#__PURE__*/ React.createElement(InfoBoxLabelCol, null, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.boldText
    }, "II ", '\n', "Other significant conditions ", '\n', "contributing to the death but", '\n', "not related to the disease or", '\n', "condition causing it.", '\n')), /*#__PURE__*/ React.createElement(InfoBoxDataCol, null, causes?.contributing?.map((cause, index)=>/*#__PURE__*/ React.createElement(CauseField, {
            style: causes?.contributing.length < 3 ? infoBoxStyles.mediumMarginTop : infoBoxStyles.smallMarginTop,
            key: index,
            cause: getCauseInfo(cause)
        }))))), /*#__PURE__*/ React.createElement(View, {
        style: generalStyles.container
    }, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.italicText
    }, "* This does not mean the mode of dying, e.g heart failure, respiratory failure. It means the disease, injury, or complication that caused death.")), deathCertFooterImg ? /*#__PURE__*/ React.createElement(SigningImage, {
        src: deathCertFooterImg
    }) : /*#__PURE__*/ React.createElement(AuthorisedAndSignSection, null), /*#__PURE__*/ React.createElement(Footer, null)));
});
export const DeathCertificatePrintout = withLanguageContext(DeathCertificatePrintoutComponent);
DeathCertificatePrintout.propTypes = {
    patientData: PropTypes.object.isRequired,
    certificateData: PropTypes.object.isRequired
};

//# sourceMappingURL=DeathCertificatePrintout.js.map