import PropTypes from 'prop-types';
import React from 'react';
import { Document, StyleSheet, View } from '@react-pdf/renderer';
import { getName } from '../patientAccessors';
import { BaseSigningSection } from './BaseSigningSection';
import { getDisplayDate } from './getDisplayDate';
import { CertificateContent, CertificateHeader, Col, Row, styles } from './Layout';
import { NOTE_TYPES } from '@tamanu/constants/notes';
import { LetterheadSection } from './LetterheadSection';
import { DataItem } from './printComponents/DataItem';
import { EncounterDetails } from './printComponents/EncounterDetails';
import { HorizontalRule } from './printComponents/HorizontalRule';
import { MultiPageHeader } from './printComponents/MultiPageHeader';
import { PatientDetailsWithBarcode } from './printComponents/PatientDetailsWithBarcode';
import { startCase } from 'lodash';
import { DoubleHorizontalRule } from './printComponents/DoubleHorizontalRule';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
const DATE_TIME_FORMAT = 'dd/MM/yyyy h:mma';
const labDetailsSectionStyles = StyleSheet.create({
    barcodeLabelText: {
        marginTop: 9
    },
    divider: {
        borderBottom: '2px solid black',
        marginVertical: '10px'
    },
    detailsContainer: {
        marginBottom: 5
    },
    heading: {
        fontFamily: 'Helvetica-Bold',
        fontSize: 11,
        fontWeight: 500,
        marginVertical: 3
    }
});
const generalStyles = StyleSheet.create({
    container: {
        marginVertical: 6
    }
});
const SectionContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.container,
        ...props
    });
const MultipleImagingRequestSigningSection = ()=>{
    return /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(Row, null, /*#__PURE__*/ React.createElement(BaseSigningSection, null)));
};
const getImagingRequestType = (imagingTypes)=>({ imagingType })=>imagingTypes[imagingType]?.label || 'Unknown';
const getAreaNote = ({ areas, areaNote })=>{
    if (areas && areas.length > 0) {
        return areas.map((area)=>area.name).join(',');
    }
    if (areaNote) {
        // there's no sensible way to key this except by array index
        // eslint-disable-next-line react/no-array-index-key
        return areaNote;
    }
    return '';
};
const ImagingRequestDetailsView = ({ imagingRequests, getLocalisation })=>{
    const notesAccessor = ({ notes })=>{
        return notes?.filter((note)=>note.noteType === NOTE_TYPES.OTHER).map((note)=>note.content).join(', ');
    };
    const imagingTypes = getLocalisation('imagingTypes') || {};
    return /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(Text, {
        style: labDetailsSectionStyles.heading
    }, "Imaging request details"), /*#__PURE__*/ React.createElement(HorizontalRule, {
        width: "0.5px"
    }), imagingRequests.map((imagingRequest, index)=>{
        return /*#__PURE__*/ React.createElement(View, {
            key: imagingRequest.id,
            style: labDetailsSectionStyles.detailsContainer
        }, /*#__PURE__*/ React.createElement(Row, null, /*#__PURE__*/ React.createElement(Col, null, /*#__PURE__*/ React.createElement(DataItem, {
            label: "Request ID",
            value: imagingRequest.displayId
        }), /*#__PURE__*/ React.createElement(DataItem, {
            label: "Priority",
            value: startCase(imagingRequest.priority)
        }), /*#__PURE__*/ React.createElement(DataItem, {
            label: "Type",
            value: getImagingRequestType(imagingTypes)(imagingRequest)
        }), /*#__PURE__*/ React.createElement(DataItem, {
            label: "Area to be imaged",
            value: getAreaNote(imagingRequest)
        }), /*#__PURE__*/ React.createElement(DataItem, {
            label: "Notes",
            value: notesAccessor(imagingRequest)
        })), /*#__PURE__*/ React.createElement(Col, null, /*#__PURE__*/ React.createElement(Row, null, /*#__PURE__*/ React.createElement(DataItem, {
            label: "Requested date & time",
            value: getDisplayDate(imagingRequest.requestedDate, DATE_TIME_FORMAT)
        }), /*#__PURE__*/ React.createElement(DataItem, {
            label: "Requested by",
            value: imagingRequest.requestedBy?.displayName
        })))), /*#__PURE__*/ React.createElement(View, {
            style: {
                marginTop: 5
            }
        }, index < imagingRequests.length - 1 ? /*#__PURE__*/ React.createElement(HorizontalRule, {
            width: "0.5px"
        }) : /*#__PURE__*/ React.createElement(DoubleHorizontalRule, null)));
    }));
};
const MultipleImagingRequestsPrintoutComponent = /*#__PURE__*/ React.memo(({ patient, imagingRequests, encounter, certificateData, getLocalisation })=>{
    const { logo } = certificateData;
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: styles.page
    }, /*#__PURE__*/ React.createElement(MultiPageHeader, {
        documentName: "Imaging request",
        patientName: getName(patient),
        patientId: patient.displayId
    }), /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        getLocalisation: getLocalisation,
        logoSrc: logo,
        letterheadConfig: certificateData,
        certificateTitle: "Imaging Request"
    }), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(PatientDetailsWithBarcode, {
        patient: patient,
        getLocalisation: getLocalisation
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(EncounterDetails, {
        encounter: encounter
    }))), /*#__PURE__*/ React.createElement(CertificateContent, null, /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(ImagingRequestDetailsView, {
        imagingRequests: imagingRequests,
        getLocalisation: getLocalisation
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(MultipleImagingRequestSigningSection, null)))));
});
export const MultipleImagingRequestsPrintout = withLanguageContext(MultipleImagingRequestsPrintoutComponent);
MultipleImagingRequestsPrintout.propTypes = {
    patient: PropTypes.object.isRequired,
    encounter: PropTypes.object.isRequired,
    imagingRequests: PropTypes.array.isRequired,
    certificateData: PropTypes.object.isRequired
};

//# sourceMappingURL=MultipleImagingRequestsPrintout.js.map