"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExpressInstrumentation = void 0;
const core_1 = require("@opentelemetry/core");
const api_1 = require("@opentelemetry/api");
const ExpressLayerType_1 = require("./enums/ExpressLayerType");
const AttributeNames_1 = require("./enums/AttributeNames");
const utils_1 = require("./utils");
const version_1 = require("./version");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const internal_types_1 = require("./internal-types");
/** Express instrumentation for OpenTelemetry */
class ExpressInstrumentation extends instrumentation_1.InstrumentationBase {
    constructor(config = {}) {
        super('@opentelemetry/instrumentation-express', version_1.VERSION, Object.assign({}, config));
    }
    setConfig(config = {}) {
        this._config = Object.assign({}, config);
    }
    getConfig() {
        return this._config;
    }
    init() {
        return [
            new instrumentation_1.InstrumentationNodeModuleDefinition('express', ['^4.0.0'], (moduleExports, moduleVersion) => {
                api_1.diag.debug(`Applying patch for express@${moduleVersion}`);
                const routerProto = moduleExports.Router;
                // patch express.Router.route
                if ((0, instrumentation_1.isWrapped)(routerProto.route)) {
                    this._unwrap(routerProto, 'route');
                }
                this._wrap(routerProto, 'route', this._getRoutePatch());
                // patch express.Router.use
                if ((0, instrumentation_1.isWrapped)(routerProto.use)) {
                    this._unwrap(routerProto, 'use');
                }
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this._wrap(routerProto, 'use', this._getRouterUsePatch());
                // patch express.Application.use
                if ((0, instrumentation_1.isWrapped)(moduleExports.application.use)) {
                    this._unwrap(moduleExports.application, 'use');
                }
                this._wrap(moduleExports.application, 'use', 
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                this._getAppUsePatch());
                return moduleExports;
            }, (moduleExports, moduleVersion) => {
                if (moduleExports === undefined)
                    return;
                api_1.diag.debug(`Removing patch for express@${moduleVersion}`);
                const routerProto = moduleExports.Router;
                this._unwrap(routerProto, 'route');
                this._unwrap(routerProto, 'use');
                this._unwrap(moduleExports.application, 'use');
            }),
        ];
    }
    /**
     * Get the patch for Router.route function
     */
    _getRoutePatch() {
        const instrumentation = this;
        return function (original) {
            return function route_trace(...args) {
                const route = original.apply(this, args);
                const layer = this.stack[this.stack.length - 1];
                instrumentation._applyPatch(layer, typeof args[0] === 'string' ? args[0] : undefined);
                return route;
            };
        };
    }
    /**
     * Get the patch for Router.use function
     */
    _getRouterUsePatch() {
        const instrumentation = this;
        return function (original) {
            return function use(...args) {
                const route = original.apply(this, args);
                const layer = this.stack[this.stack.length - 1];
                instrumentation._applyPatch(layer, typeof args[0] === 'string' ? args[0] : undefined);
                return route;
            };
        };
    }
    /**
     * Get the patch for Application.use function
     */
    _getAppUsePatch() {
        const instrumentation = this;
        return function (original) {
            return function use(...args) {
                const route = original.apply(this, args);
                const layer = this._router.stack[this._router.stack.length - 1];
                instrumentation._applyPatch.call(instrumentation, layer, typeof args[0] === 'string' ? args[0] : undefined);
                return route;
            };
        };
    }
    /** Patch each express layer to create span and propagate context */
    _applyPatch(layer, layerPath) {
        const instrumentation = this;
        // avoid patching multiple times the same layer
        if (layer[internal_types_1.kLayerPatched] === true)
            return;
        layer[internal_types_1.kLayerPatched] = true;
        this._wrap(layer, 'handle', (original) => {
            if (original.length === 4)
                return original;
            return function (req, res) {
                (0, utils_1.storeLayerPath)(req, layerPath);
                const route = req[internal_types_1._LAYERS_STORE_PROPERTY]
                    .filter(path => path !== '/' && path !== '/*')
                    .join('');
                const attributes = {
                    [semantic_conventions_1.SemanticAttributes.HTTP_ROUTE]: route.length > 0 ? route : '/',
                };
                const metadata = (0, utils_1.getLayerMetadata)(layer, layerPath);
                const type = metadata.attributes[AttributeNames_1.AttributeNames.EXPRESS_TYPE];
                // Rename the root http span in case we haven't done it already
                // once we reach the request handler
                const rpcMetadata = (0, core_1.getRPCMetadata)(api_1.context.active());
                if (metadata.attributes[AttributeNames_1.AttributeNames.EXPRESS_TYPE] ===
                    ExpressLayerType_1.ExpressLayerType.REQUEST_HANDLER &&
                    (rpcMetadata === null || rpcMetadata === void 0 ? void 0 : rpcMetadata.type) === core_1.RPCType.HTTP) {
                    const name = instrumentation._getSpanName({
                        request: req,
                        route,
                    }, `${req.method} ${route.length > 0 ? route : '/'}`);
                    rpcMetadata.span.updateName(name);
                }
                // verify against the config if the layer should be ignored
                if ((0, utils_1.isLayerIgnored)(metadata.name, type, instrumentation._config)) {
                    if (type === ExpressLayerType_1.ExpressLayerType.MIDDLEWARE) {
                        req[internal_types_1._LAYERS_STORE_PROPERTY].pop();
                    }
                    return original.apply(this, arguments);
                }
                if (api_1.trace.getSpan(api_1.context.active()) === undefined) {
                    return original.apply(this, arguments);
                }
                const spanName = instrumentation._getSpanName({
                    request: req,
                    layerType: type,
                    route,
                }, metadata.name);
                const span = instrumentation.tracer.startSpan(spanName, {
                    attributes: Object.assign(attributes, metadata.attributes),
                });
                if (instrumentation.getConfig().requestHook) {
                    (0, instrumentation_1.safeExecuteInTheMiddle)(() => instrumentation.getConfig().requestHook(span, {
                        request: req,
                        layerType: type,
                        route,
                    }), e => {
                        if (e) {
                            api_1.diag.error('express instrumentation: request hook failed', e);
                        }
                    }, true);
                }
                let spanHasEnded = false;
                if (metadata.attributes[AttributeNames_1.AttributeNames.EXPRESS_TYPE] !==
                    ExpressLayerType_1.ExpressLayerType.MIDDLEWARE) {
                    span.end();
                    spanHasEnded = true;
                }
                // listener for response.on('finish')
                const onResponseFinish = () => {
                    if (spanHasEnded === false) {
                        spanHasEnded = true;
                        span.end();
                    }
                };
                // verify we have a callback
                const args = Array.from(arguments);
                const callbackIdx = args.findIndex(arg => typeof arg === 'function');
                const newContext = (rpcMetadata === null || rpcMetadata === void 0 ? void 0 : rpcMetadata.type) === core_1.RPCType.HTTP
                    ? (0, core_1.setRPCMetadata)(api_1.context.active(), Object.assign(rpcMetadata, { route: route }))
                    : api_1.context.active();
                if (callbackIdx >= 0) {
                    arguments[callbackIdx] = function () {
                        var _a;
                        if (spanHasEnded === false) {
                            spanHasEnded = true;
                            (_a = req.res) === null || _a === void 0 ? void 0 : _a.removeListener('finish', onResponseFinish);
                            span.end();
                        }
                        if (!(req.route && arguments[0] instanceof Error)) {
                            req[internal_types_1._LAYERS_STORE_PROPERTY].pop();
                        }
                        const callback = args[callbackIdx];
                        return api_1.context.bind(newContext, callback).apply(this, arguments);
                    };
                }
                const result = original.apply(this, arguments);
                /**
                 * At this point if the callback wasn't called, that means either the
                 * layer is asynchronous (so it will call the callback later on) or that
                 * the layer directly end the http response, so we'll hook into the "finish"
                 * event to handle the later case.
                 */
                if (!spanHasEnded) {
                    res.once('finish', onResponseFinish);
                }
                return result;
            };
        });
    }
    _getSpanName(info, defaultName) {
        var _a;
        const hook = this.getConfig().spanNameHook;
        if (!(hook instanceof Function)) {
            return defaultName;
        }
        try {
            return (_a = hook(info, defaultName)) !== null && _a !== void 0 ? _a : defaultName;
        }
        catch (err) {
            api_1.diag.error('express instrumentation: error calling span name rewrite hook', err);
            return defaultName;
        }
    }
}
exports.ExpressInstrumentation = ExpressInstrumentation;
//# sourceMappingURL=instrumentation.js.map