"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.shouldNotTraceServerCall = exports.serverStreamAndBidiHandler = exports.clientStreamAndUnaryHandler = void 0;
const api_1 = require("@opentelemetry/api");
const utils_1 = require("../utils");
const AttributeNames_1 = require("../enums/AttributeNames");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const status_code_1 = require("../status-code");
const clientStreamAndUnaryHandler = function (span, call, callback, original, self) {
    function patchedCallback(err, value, trailer, flags) {
        if (err) {
            if (err.code) {
                span.setStatus({
                    code: (0, utils_1._grpcStatusCodeToOpenTelemetryStatusCode)(err.code),
                    message: err.message,
                });
                span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, err.code);
            }
            span.setAttributes({
                [AttributeNames_1.AttributeNames.GRPC_ERROR_NAME]: err.name,
                [AttributeNames_1.AttributeNames.GRPC_ERROR_MESSAGE]: err.message,
            });
        }
        else {
            span.setStatus({ code: api_1.SpanStatusCode.UNSET });
            span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, status_code_1.GRPC_STATUS_CODE_OK);
        }
        span.addEvent('received');
        // end the span
        span.end();
        return callback(err, value, trailer, flags);
    }
    api_1.context.bind(api_1.context.active(), call);
    return original.call(self, call, patchedCallback);
};
exports.clientStreamAndUnaryHandler = clientStreamAndUnaryHandler;
const serverStreamAndBidiHandler = function (span, call, original, self) {
    let spanEnded = false;
    const endSpan = () => {
        if (!spanEnded) {
            spanEnded = true;
            span.end();
        }
    };
    api_1.context.bind(api_1.context.active(), call);
    call.on('finish', () => {
        span.setStatus((0, utils_1._grpcStatusCodeToSpanStatus)(call.status.code));
        span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, call.status.code);
        // if there is an error, span will be ended on error event, otherwise end it here
        if (call.status.code === 0) {
            span.addEvent('finished');
            endSpan();
        }
    });
    call.on('error', (err) => {
        span.setStatus({
            code: (0, utils_1._grpcStatusCodeToOpenTelemetryStatusCode)(err.code),
            message: err.message,
        });
        span.addEvent('finished with error');
        span.setAttributes({
            [AttributeNames_1.AttributeNames.GRPC_ERROR_NAME]: err.name,
            [AttributeNames_1.AttributeNames.GRPC_ERROR_MESSAGE]: err.message,
        });
        if (err.code != null) {
            span.setAttribute(semantic_conventions_1.SemanticAttributes.RPC_GRPC_STATUS_CODE, err.code);
        }
        endSpan();
    });
    return original.call(self, call);
};
exports.serverStreamAndBidiHandler = serverStreamAndBidiHandler;
/**
 * Returns true if the server call should not be traced.
 */
const shouldNotTraceServerCall = function (call, name) {
    const parsedName = name.split('/');
    return (0, utils_1._methodIsIgnored)(parsedName[parsedName.length - 1] || name, this.getConfig().ignoreGrpcMethods);
};
exports.shouldNotTraceServerCall = shouldNotTraceServerCall;
//# sourceMappingURL=serverUtils.js.map