"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    authMiddleware: function() {
        return authMiddleware;
    },
    centralServerLogin: function() {
        return centralServerLogin;
    },
    comparePassword: function() {
        return comparePassword;
    },
    getToken: function() {
        return getToken;
    },
    loginHandler: function() {
        return loginHandler;
    },
    refreshHandler: function() {
        return refreshHandler;
    }
});
const _api = require("@opentelemetry/api");
const _jsonwebtoken = require("jsonwebtoken");
const _bcrypt = require("bcrypt");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _util = require("util");
const _crypto = /*#__PURE__*/ _interop_require_default(require("crypto"));
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/shared/errors");
const _logging = require("@tamanu/shared/services/logging");
const _rolesToPermissions = require("@tamanu/shared/permissions/rolesToPermissions");
const _sync = require("../sync");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const { tokenDuration, secret } = _config.default.auth;
// regenerate the secret key whenever the server restarts.
// this will invalidate all current tokens, but they're meant to expire fairly quickly anyway.
const jwtSecretKey = secret || _crypto.default.randomUUID();
const sign = (0, _util.promisify)(_jsonwebtoken.sign);
const verify = (0, _util.promisify)(_jsonwebtoken.verify);
async function getToken(user, expiresIn = tokenDuration) {
    return sign({
        userId: user.id
    }, jwtSecretKey, {
        expiresIn
    });
}
async function comparePassword(user, password) {
    try {
        const passwordHash = user && user.password;
        // do the password comparison even if the user is invalid so
        // that the login check doesn't reveal whether a user exists or not
        const passwordMatch = await (0, _bcrypt.compare)(password, passwordHash || 'invalid-hash');
        return user && passwordMatch;
    } catch (e) {
        return false;
    }
}
async function centralServerLogin(models, email, password, deviceId) {
    // try logging in to central server
    const centralServer = new _sync.CentralServerConnection({
        deviceId
    });
    const response = await centralServer.fetch('login', {
        awaitConnection: false,
        retryAuth: false,
        method: 'POST',
        body: {
            email,
            password,
            deviceId
        },
        backoff: {
            maxAttempts: 1
        }
    });
    // we've logged in as a valid central user - update local database to match
    const { user, localisation, settings } = response;
    const { id, ...userDetails } = user;
    await models.User.sequelize.transaction(async ()=>{
        await models.User.upsert({
            id,
            ...userDetails,
            password,
            deletedAt: null
        });
        await models.UserLocalisationCache.upsert({
            userId: id,
            localisation: JSON.stringify(localisation),
            deletedAt: null
        });
    });
    return {
        central: true,
        user,
        localisation,
        settings
    };
}
async function localLogin(models, email, password) {
    // some other error in communicating with central server, revert to local login
    const user = await models.User.getForAuthByEmail(email);
    const passwordMatch = await comparePassword(user, password);
    if (!passwordMatch) {
        throw new _errors.BadAuthenticationError('Incorrect username or password, please try again');
    }
    const localisation = await models.UserLocalisationCache.getLocalisation({
        where: {
            userId: user.id
        },
        order: [
            [
                'createdAt',
                'DESC'
            ]
        ]
    });
    return {
        central: false,
        user,
        localisation
    };
}
async function centralServerLoginWithLocalFallback(models, email, password, deviceId) {
    // always log in locally when testing
    if (process.env.NODE_ENV === 'test') {
        return localLogin(models, email, password);
    }
    try {
        return await centralServerLogin(models, email, password, deviceId);
    } catch (e) {
        if (e.name === 'BadAuthenticationError') {
            // actual bad credentials server-side
            throw new _errors.BadAuthenticationError('Incorrect username or password, please try again');
        }
        // if it is forbidden error when login to central server,
        // throw the error instead of proceeding to local login
        if (e.centralServerResponse?.status === 403 && e.centralServerResponse?.body?.error) {
            throw e.centralServerResponse.body.error;
        }
        _logging.log.warn(`centralServerLoginWithLocalFallback: central server login failed: ${e}`);
        return localLogin(models, email, password);
    }
}
async function loginHandler(req, res, next) {
    const { body, models, deviceId, settings } = req;
    const { email, password } = body;
    // no permission needed for login
    req.flagPermissionChecked();
    try {
        const { central, user, localisation } = await centralServerLoginWithLocalFallback(models, email, password, deviceId);
        const [facility, permissions, token, role] = await Promise.all([
            models.Facility.findByPk(_config.default.serverFacilityId),
            (0, _rolesToPermissions.getPermissionsForRoles)(models, user.role),
            getToken(user),
            models.Role.findByPk(user.role)
        ]);
        res.send({
            token,
            central,
            localisation,
            permissions,
            role: role?.forResponse() ?? null,
            server: {
                facility: facility?.forResponse() ?? null
            },
            settings: await settings.getFrontEndSettings()
        });
    } catch (e) {
        next(e);
    }
}
async function refreshHandler(req, res) {
    const { user } = req;
    // Run after auth middleware, requires valid token but no other permission
    req.flagPermissionChecked();
    const token = await getToken(user);
    res.send({
        token
    });
}
function decodeToken(token) {
    return verify(token, jwtSecretKey);
}
async function getUserFromToken(request) {
    const { models, headers } = request;
    const authHeader = headers.authorization || '';
    if (!authHeader) return null;
    const bearer = authHeader.match(/Bearer (\S*)/);
    if (!bearer) {
        throw new _errors.BadAuthenticationError('Missing auth token header');
    }
    const token = bearer[1];
    try {
        const { userId } = await decodeToken(token);
        const user = await models.User.findByPk(userId);
        if (user.visibilityStatus !== _constants.VISIBILITY_STATUSES.CURRENT) {
            throw new Error('User is not visible to the system'); // will be caught immediately
        }
        return user;
    } catch (e) {
        throw new _errors.BadAuthenticationError('Your session has expired or is invalid. Please log in again.');
    }
}
const authMiddleware = async (req, res, next)=>{
    try {
        // eslint-disable-next-line require-atomic-updates
        req.user = await getUserFromToken(req);
        req.getLocalisation = async ()=>req.models.UserLocalisationCache.getLocalisation({
                where: {
                    userId: req.user.id
                },
                order: [
                    [
                        'createdAt',
                        'DESC'
                    ]
                ]
            });
        const spanAttributes = req.user ? {
            'enduser.id': req.user.id,
            'enduser.role': req.user.role
        } : {};
        // eslint-disable-next-line no-unused-expressions
        _api.trace.getActiveSpan()?.setAttributes(spanAttributes);
        _api.context.with(_api.propagation.setBaggage(_api.context.active(), _api.propagation.createBaggage(spanAttributes)), ()=>next());
    } catch (e) {
        next(e);
    }
};

//# sourceMappingURL=auth.js.map