"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "ImagingResult", {
    enumerable: true,
    get: function() {
        return ImagingResult;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _Model = require("./Model");
const _buildEncounterLinkedSyncFilter = require("./buildEncounterLinkedSyncFilter");
const _dateTimeTypes = require("./dateTimeTypes");
const _dateTime = require("../utils/dateTime");
const _buildPatientLinkedLookupFilter = require("./buildPatientLinkedLookupFilter");
let ImagingResult = class ImagingResult extends _Model.Model {
    static init(options) {
        super.init({
            id: {
                type: _sequelize.DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: _sequelize.Sequelize.fn('uuid_generate_v4')
            },
            visibilityStatus: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: 'current'
            },
            completedAt: (0, _dateTimeTypes.dateTimeType)('completedAt', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            description: {
                type: _sequelize.DataTypes.TEXT
            },
            externalCode: _sequelize.DataTypes.TEXT
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            validate: {
                mustHaveImagingRequest () {
                    if (!this.imagingRequestId) {
                        throw new _errors.InvalidOperationError('An imaging result must be associated with an imaging request.');
                    }
                }
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'request',
            'completedBy'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.ImagingRequest, {
            foreignKey: 'imagingRequestId',
            as: 'request'
        });
        this.belongsTo(models.User, {
            foreignKey: 'completedById',
            as: 'completedBy'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'imaging_requests',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildPatientLinkedLookupFilter.buildEncounterPatientIdSelect)(this),
            joins: (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilterJoins)([
                this.tableName,
                'imaging_requests',
                'encounters'
            ])
        };
    }
};

//# sourceMappingURL=ImagingResult.js.map