"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "SyncSession", {
    enumerable: true,
    get: function() {
        return SyncSession;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _logging = require("@tamanu/shared/services/logging");
let SyncSession = class SyncSession extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            startTime: {
                type: _sequelize.DataTypes.DATE
            },
            lastConnectionTime: {
                type: _sequelize.DataTypes.DATE
            },
            snapshotStartedAt: {
                type: _sequelize.DataTypes.DATE
            },
            snapshotCompletedAt: {
                type: _sequelize.DataTypes.DATE
            },
            persistCompletedAt: {
                type: _sequelize.DataTypes.DATE
            },
            completedAt: {
                type: _sequelize.DataTypes.DATE
            },
            startedAtTick: {
                type: _sequelize.DataTypes.BIGINT
            },
            pullSince: {
                type: _sequelize.DataTypes.BIGINT
            },
            pullUntil: {
                type: _sequelize.DataTypes.BIGINT
            },
            errors: {
                type: _sequelize.DataTypes.ARRAY(_sequelize.DataTypes.TEXT)
            },
            debugInfo: {
                type: _sequelize.DataTypes.JSON
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.DO_NOT_SYNC
        });
    }
    /**
   * Set business-logic parameters to the sync session.
   * As this writes to the table, it may contend locks.
   * For debugging information, use addDebugInfo() instead.
   */ static async setParameters(id, params) {
        await this.sequelize.query(`
      UPDATE "sync_sessions"
      SET "debug_info" = (COALESCE("debug_info"::jsonb, '{}'::jsonb) || $data::jsonb)::json
      WHERE "id" = $id
      `, {
            bind: {
                id,
                data: JSON.stringify(params)
            }
        });
    }
    /**
   * Add information useful for debugging sync to the session.
   * To avoid contending locks, this will drop the data if it
   * can't obtain the write. For business-logic parameters, use
   * setParameters() instead.
   */ static async addDebugInfo(id, info) {
        // the SKIP LOCKED means we don't lock the sync_sessions table's row
        // if it's already contended. but that also means that we might lose
        // some debug info in some cases. so let's debug log it as well.
        _logging.log.debug('Sync session debug', {
            id,
            ...info
        });
        const session = await this.findOne({
            where: {
                id
            },
            skipLocked: true
        });
        await session?.update({
            debugInfo: {
                ...session.debugInfo,
                ...info
            }
        });
    }
    async markErrored(error) {
        const errors = this.errors || [];
        await this.update({
            errors: [
                ...errors,
                error
            ],
            completedAt: new Date()
        });
    }
    static async markSessionErrored(id, error) {
        const session = await this.findByPk(id);
        await session.markErrored(error);
    }
};

//# sourceMappingURL=SyncSession.js.map