import { DataTypes, Sequelize } from 'sequelize';
import { SYNC_DIRECTIONS } from '@tamanu/constants';
import { InvalidOperationError } from '../errors';
import { Model } from './Model';
import { buildEncounterLinkedSyncFilter, buildEncounterLinkedSyncFilterJoins } from './buildEncounterLinkedSyncFilter';
import { dateTimeType } from './dateTimeTypes';
import { getCurrentDateTimeString } from '../utils/dateTime';
import { buildEncounterPatientIdSelect } from './buildPatientLinkedLookupFilter';
export class ImagingResult extends Model {
    static init(options) {
        super.init({
            id: {
                type: DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: Sequelize.fn('uuid_generate_v4')
            },
            visibilityStatus: {
                type: DataTypes.STRING,
                allowNull: false,
                defaultValue: 'current'
            },
            completedAt: dateTimeType('completedAt', {
                allowNull: false,
                defaultValue: getCurrentDateTimeString
            }),
            description: {
                type: DataTypes.TEXT
            },
            externalCode: DataTypes.TEXT
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.BIDIRECTIONAL,
            validate: {
                mustHaveImagingRequest () {
                    if (!this.imagingRequestId) {
                        throw new InvalidOperationError('An imaging result must be associated with an imaging request.');
                    }
                }
            }
        });
    }
    static getListReferenceAssociations() {
        return [
            'request',
            'completedBy'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.ImagingRequest, {
            foreignKey: 'imagingRequestId',
            as: 'request'
        });
        this.belongsTo(models.User, {
            foreignKey: 'completedById',
            as: 'completedBy'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return buildEncounterLinkedSyncFilter([
            this.tableName,
            'imaging_requests',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: buildEncounterPatientIdSelect(this),
            joins: buildEncounterLinkedSyncFilterJoins([
                this.tableName,
                'imaging_requests',
                'encounters'
            ])
        };
    }
}

//# sourceMappingURL=ImagingResult.js.map