"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeSDK = void 0;
const api_1 = require("@opentelemetry/api");
const instrumentation_1 = require("@opentelemetry/instrumentation");
const resources_1 = require("@opentelemetry/resources");
const sdk_metrics_1 = require("@opentelemetry/sdk-metrics");
const sdk_trace_base_1 = require("@opentelemetry/sdk-trace-base");
const sdk_trace_node_1 = require("@opentelemetry/sdk-trace-node");
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const TracerProviderWithEnvExporter_1 = require("./TracerProviderWithEnvExporter");
class NodeSDK {
    /**
     * Create a new NodeJS SDK instance
     */
    constructor(configuration = {}) {
        var _a, _b, _c, _d;
        this._resource = (_a = configuration.resource) !== null && _a !== void 0 ? _a : new resources_1.Resource({});
        this._resourceDetectors = (_b = configuration.resourceDetectors) !== null && _b !== void 0 ? _b : [resources_1.envDetector, resources_1.processDetector];
        this._serviceName = configuration.serviceName;
        this._autoDetectResources = (_c = configuration.autoDetectResources) !== null && _c !== void 0 ? _c : true;
        if (configuration.spanProcessor || configuration.traceExporter) {
            const tracerProviderConfig = {};
            if (configuration.sampler) {
                tracerProviderConfig.sampler = configuration.sampler;
            }
            if (configuration.spanLimits) {
                tracerProviderConfig.spanLimits = configuration.spanLimits;
            }
            const spanProcessor = (_d = configuration.spanProcessor) !== null && _d !== void 0 ? _d : new sdk_trace_base_1.BatchSpanProcessor(configuration.traceExporter);
            this.configureTracerProvider(tracerProviderConfig, spanProcessor, configuration.contextManager, configuration.textMapPropagator);
        }
        if (configuration.metricReader || configuration.views) {
            const meterProviderConfig = {};
            if (configuration.metricReader) {
                meterProviderConfig.reader = configuration.metricReader;
            }
            if (configuration.views) {
                meterProviderConfig.views = configuration.views;
            }
            this.configureMeterProvider(meterProviderConfig);
        }
        let instrumentations = [];
        if (configuration.instrumentations) {
            instrumentations = configuration.instrumentations;
        }
        this._instrumentations = instrumentations;
    }
    /** Set configurations required to register a NodeTracerProvider */
    configureTracerProvider(tracerConfig, spanProcessor, contextManager, textMapPropagator) {
        this._tracerProviderConfig = {
            tracerConfig,
            spanProcessor,
            contextManager,
            textMapPropagator,
        };
    }
    /** Set configurations needed to register a MeterProvider */
    configureMeterProvider(config) {
        // nothing is set yet, we can set config and return.
        if (this._meterProviderConfig == null) {
            this._meterProviderConfig = config;
            return;
        }
        // make sure we do not override existing views with other views.
        if (this._meterProviderConfig.views != null && config.views != null) {
            throw new Error('Views passed but Views have already been configured.');
        }
        // set views, but make sure we do not override existing views with null/undefined.
        if (config.views != null) {
            this._meterProviderConfig.views = config.views;
        }
        // make sure we do not override existing reader with another reader.
        if (this._meterProviderConfig.reader != null && config.reader != null) {
            throw new Error('MetricReader passed but MetricReader has already been configured.');
        }
        // set reader, but make sure we do not override existing reader with null/undefined.
        if (config.reader != null) {
            this._meterProviderConfig.reader = config.reader;
        }
    }
    /** Detect resource attributes */
    async detectResources() {
        const internalConfig = {
            detectors: this._resourceDetectors,
        };
        this.addResource(await (0, resources_1.detectResources)(internalConfig));
    }
    /** Manually add a resource */
    addResource(resource) {
        this._resource = this._resource.merge(resource);
    }
    /**
     * Once the SDK has been configured, call this method to construct SDK components and register them with the OpenTelemetry API.
     */
    async start() {
        var _a, _b, _c, _d, _e;
        if (this._autoDetectResources) {
            await this.detectResources();
        }
        this._resource = this._serviceName === undefined
            ? this._resource
            : this._resource.merge(new resources_1.Resource({ [semantic_conventions_1.SemanticResourceAttributes.SERVICE_NAME]: this._serviceName }));
        const Provider = this._tracerProviderConfig ? sdk_trace_node_1.NodeTracerProvider : TracerProviderWithEnvExporter_1.TracerProviderWithEnvExporters;
        const tracerProvider = new Provider(Object.assign(Object.assign({}, (_a = this._tracerProviderConfig) === null || _a === void 0 ? void 0 : _a.tracerConfig), { resource: this._resource }));
        this._tracerProvider = tracerProvider;
        if (this._tracerProviderConfig) {
            tracerProvider.addSpanProcessor(this._tracerProviderConfig.spanProcessor);
        }
        tracerProvider.register({
            contextManager: (_b = this._tracerProviderConfig) === null || _b === void 0 ? void 0 : _b.contextManager,
            propagator: (_c = this._tracerProviderConfig) === null || _c === void 0 ? void 0 : _c.textMapPropagator,
        });
        if (this._meterProviderConfig) {
            const meterProvider = new sdk_metrics_1.MeterProvider({
                resource: this._resource,
                views: (_e = (_d = this._meterProviderConfig) === null || _d === void 0 ? void 0 : _d.views) !== null && _e !== void 0 ? _e : [],
            });
            if (this._meterProviderConfig.reader) {
                meterProvider.addMetricReader(this._meterProviderConfig.reader);
            }
            this._meterProvider = meterProvider;
            api_1.metrics.setGlobalMeterProvider(meterProvider);
        }
        (0, instrumentation_1.registerInstrumentations)({
            instrumentations: this._instrumentations,
        });
    }
    shutdown() {
        const promises = [];
        if (this._tracerProvider) {
            promises.push(this._tracerProvider.shutdown());
        }
        if (this._meterProvider) {
            promises.push(this._meterProvider.shutdown());
        }
        return (Promise.all(promises)
            // return void instead of the array from Promise.all
            .then(() => {
        }));
    }
}
exports.NodeSDK = NodeSDK;
//# sourceMappingURL=sdk.js.map