"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "importSurvey", {
    enumerable: true,
    get: function() {
        return importSurvey;
    }
});
const _xlsx = require("xlsx");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _importRows = require("../importer/importRows");
const _readSurveyQuestions = require("./readSurveyQuestions");
const _validation = require("./validation");
const _vitalsValidation = require("./vitalsValidation");
function readSurveyInfo(workbook, surveyInfo) {
    const { sheetName, surveyType, code } = surveyInfo;
    const surveyRecord = {
        model: 'Survey',
        sheetRow: -2,
        values: surveyInfo
    };
    // don't bother looking for the sheet of questions from obsoleted surveys
    // (it may not even exist in the doc, and that's fine)
    if (surveyType === _constants.SURVEY_TYPES.OBSOLETE) {
        return [
            surveyRecord
        ];
    }
    // Strip some characters from workbook names before trying to find them
    // (this mirrors the punctuation stripping that node-xlsx does internally)
    const worksheet = workbook.Sheets[sheetName.replace(/['"]/g, '')] || workbook.Sheets[code];
    if (!worksheet) {
        const keys = Object.keys(workbook.Sheets);
        throw new _errors.ImporterMetadataError(`Sheet named "${sheetName}" was not found in the workbook. (found: ${keys})`);
    }
    const data = _xlsx.utils.sheet_to_json(worksheet);
    const questionRecords = (0, _readSurveyQuestions.readSurveyQuestions)(data, surveyInfo);
    (0, _validation.ensureRequiredQuestionsPresent)(surveyInfo, questionRecords);
    return [
        surveyRecord,
        ...questionRecords
    ];
}
async function importSurvey(context, workbook, surveyInfo) {
    var _surveyInfo;
    const { sheetName, surveyType, programId } = surveyInfo;
    if (surveyType === _constants.SURVEY_TYPES.VITALS) {
        await (0, _validation.validateVitalsSurvey)(context, surveyInfo);
    }
    (_surveyInfo = surveyInfo).notifiable ?? (_surveyInfo.notifiable = false);
    surveyInfo.notifyEmailAddresses = (surveyInfo.notifyEmailAddresses ?? '').split(',').map((email)=>email.trim()).filter(Boolean);
    const records = readSurveyInfo(workbook, surveyInfo);
    const stats = (0, _vitalsValidation.validateProgramDataElementRecords)(records, {
        context,
        sheetName
    });
    return (0, _importRows.importRows)(context, {
        sheetName,
        rows: records,
        stats
    }, {
        models: context.models,
        programId
    });
}

//# sourceMappingURL=importSurvey.js.map