"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "PatientTelegramCommunicationProcessor", {
    enumerable: true,
    get: function() {
        return PatientTelegramCommunicationProcessor;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _constants = require("@tamanu/constants");
const _tasks = require("@tamanu/shared/tasks");
const _logging = require("@tamanu/shared/services/logging");
const _errors = require("@tamanu/shared/errors");
const _utils = require("@tamanu/shared/utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
let PatientTelegramCommunicationProcessor = class PatientTelegramCommunicationProcessor extends _tasks.ScheduledTask {
    getName() {
        return 'PatientTelegramCommunicationProcessor';
    }
    async countQueue() {
        const { PatientCommunication } = this.context.store.models;
        return PatientCommunication.countPendingMessages(_constants.PATIENT_COMMUNICATION_CHANNELS.TELEGRAM);
    }
    async run() {
        const { PatientCommunication } = this.context.store.models;
        const toProcess = await this.countQueue();
        if (toProcess === 0) return;
        const { batchSize, batchSleepAsyncDurationInMilliseconds } = this.config;
        // Make sure these exist, else they will prevent the script from working
        if (!batchSize || !batchSleepAsyncDurationInMilliseconds) {
            throw new _errors.InvalidConfigError('batchSize and batchSleepAsyncDurationInMilliseconds must be set for PatientTelegramCommunicationProcessor');
        }
        const batchCount = Math.ceil(toProcess / batchSize);
        _logging.log.info('Running batched sending reminder message to patient via telegram', {
            recordCount: toProcess,
            batchCount,
            batchSize
        });
        for(let i = 0; i < batchCount; i++){
            const communications = await PatientCommunication.getPendingMessages(_constants.PATIENT_COMMUNICATION_CHANNELS.TELEGRAM, {
                limit: batchSize
            });
            for (const communication of communications){
                const plainCommunication = communication.get({
                    plain: true
                });
                const result = await this.context.telegramBotService.sendMessage(plainCommunication.destination, communication.content);
                if (result.error) {
                    _logging.log.error('Sending message via telegram failed', {
                        communicationId: plainCommunication.id,
                        error: result.error
                    });
                    if (result.shouldRetry) {
                        await communication.update({
                            retryCount: plainCommunication.retryCount + 1,
                            error: result.error
                        });
                        continue;
                    }
                }
                await communication.update({
                    status: result.status,
                    error: result.error
                });
            }
            await (0, _utils.sleepAsync)(batchSleepAsyncDurationInMilliseconds);
        }
    }
    constructor(context){
        const conf = _config.default.schedules.patientTelegramCommunicationProcessor;
        const { schedule, jitterTime, enabled } = conf;
        super(schedule, _logging.log, jitterTime, enabled);
        this.config = conf;
        this.context = context;
    }
};

//# sourceMappingURL=PatientTelegramCommunicationProcessor.js.map