"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "dataGenerator", {
    enumerable: true,
    get: function() {
        return dataGenerator;
    }
});
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _upperFirst = /*#__PURE__*/ _interop_require_default(require("lodash/upperFirst"));
const _constants = require("@tamanu/constants");
const _Location = require("../models/Location");
const _dateTime = require("../utils/dateTime");
const _utilities = require("./utilities");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const reportColumnTemplate = [
    {
        title: 'Patient First Name',
        accessor: (data)=>data.patient.firstName
    },
    {
        title: 'Patient Last Name',
        accessor: (data)=>data.patient.lastName
    },
    {
        title: 'Patient ID',
        accessor: (data)=>data.patient.displayId
    },
    {
        title: 'Sex',
        accessor: (data)=>data.patient.sex
    },
    {
        title: 'Village',
        accessor: (data)=>data.patient.village.name
    },
    {
        title: 'Date of Birth',
        accessor: (data)=>(0, _dateTime.format)(data.patient.dateOfBirth, 'dd/MM/yyyy')
    },
    {
        title: 'Age',
        accessor: (data)=>(0, _dateTime.ageInYears)(data.patient.dateOfBirth)
    },
    {
        title: 'Patient Type',
        accessor: (data)=>data.patientBillingType?.name
    },
    {
        title: 'Admitting Clinician',
        accessor: (data)=>data.examiner?.displayName
    },
    {
        title: 'Admission Date',
        accessor: (data)=>(0, _dateTime.format)(data.startDate, 'dd/MM/yyyy h:mm:ss a')
    },
    {
        title: 'Discharge Date',
        accessor: (data)=>data.endDate && (0, _dateTime.format)(data.endDate, 'dd/MM/yyyy h:mm:ss a')
    },
    {
        title: 'Location',
        accessor: (data)=>data.locationHistoryString
    },
    {
        title: 'Department',
        accessor: (data)=>data.departmentHistoryString
    },
    {
        title: 'Primary diagnoses',
        accessor: (data)=>data.primaryDiagnoses
    },
    {
        title: 'Secondary diagnoses',
        accessor: (data)=>data.secondaryDiagnoses
    }
];
function parametersToSqlWhere(parameters) {
    const { fromDate, toDate, practitioner, patientBillingType } = parameters;
    const queryFromDate = (0, _dateTime.toDateTimeString)((0, _datefns.startOfDay)(fromDate ? (0, _datefns.parseISO)(fromDate) : (0, _datefns.subDays)(new Date(), 30)));
    const queryToDate = toDate && (0, _dateTime.toDateTimeString)((0, _datefns.endOfDay)((0, _datefns.parseISO)(toDate)));
    return {
        encounterType: _constants.ENCOUNTER_TYPES.ADMISSION,
        ...patientBillingType && {
            patientBillingTypeId: patientBillingType
        },
        ...practitioner && {
            examinerId: practitioner
        },
        startDate: {
            [_sequelize.Op.gte]: queryFromDate,
            ...queryToDate && {
                [_sequelize.Op.lte]: queryToDate
            }
        }
    };
}
const stringifyDiagnoses = (diagnoses, shouldBePrimary)=>diagnoses.filter(({ isPrimary })=>isPrimary === shouldBePrimary).map(({ Diagnosis })=>`${Diagnosis.code} ${Diagnosis.name}`).join('; ');
const getAllNotes = async (models, encounterIds)=>{
    const locationChangeNotes = await models.Note.findAll({
        where: {
            content: {
                [_sequelize.Op.like]: 'Changed location from%'
            },
            recordId: encounterIds,
            noteType: _constants.NOTE_TYPES.SYSTEM,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    const departmentChangeNotes = await models.Note.findAll({
        where: {
            content: {
                [_sequelize.Op.like]: 'Changed department from%'
            },
            recordId: encounterIds,
            noteType: _constants.NOTE_TYPES.SYSTEM,
            visibilityStatus: _constants.VISIBILITY_STATUSES.CURRENT
        }
    });
    return {
        locationChangeNotes,
        departmentChangeNotes
    };
};
// Note - hard to figure out departments with a ' to ' in them:
// Changed department from Department x to Department to be
// Could be: "Department x"/"Department to be" or "Department x to Department"/"be"
const locationExtractorPattern = /^Changed location from (?<from>.*) to (?<to>.*)/;
const departmentExtractorPattern = /^Changed department from (?<from>.*) to (?<to>.*)/;
const patternsForPlaceTypes = {
    department: departmentExtractorPattern,
    location: locationExtractorPattern
};
const getPlaceHistoryFromNotes = (changeNotes, encounterData, placeType)=>{
    const relevantNotes = changeNotes.filter(({ recordId })=>recordId === encounterData.id).sort(({ date })=>date);
    if (!relevantNotes.length) {
        const { [placeType]: place, startDate } = encounterData;
        const placeName = _Location.Location.formatFullLocationName(place);
        return [
            {
                to: placeName,
                date: startDate
            }
        ];
    }
    const matcher = patternsForPlaceTypes[placeType];
    const { groups: { from } } = relevantNotes[0].content.match(matcher);
    const history = [
        {
            to: from,
            date: encounterData.startDate
        },
        ...relevantNotes.map(({ content, date })=>{
            const { groups: { to } } = content.match(matcher);
            return {
                to,
                date
            };
        })
    ];
    return history;
};
const formatHistory = (history, placeType)=>{
    const items = history.map(({ to, date })=>{
        return `${to} (${(0, _upperFirst.default)(placeType)} assigned: ${(0, _dateTime.format)(date, 'dd/MM/yy h:mm a')})`;
    });
    return items.join('; ');
};
const filterResults = async (models, results, parameters)=>{
    const { locationGroup, department } = parameters;
    const locations = locationGroup && await models.Location.findAll({
        where: {
            locationGroupId: locationGroup
        }
    });
    const { name: locationGroupName } = locationGroup ? await models.LocationGroup.findOne({
        where: {
            id: locationGroup
        }
    }) : {};
    const locationNames = locations?.map(({ name })=>name);
    const { name: requiredDepartment } = await models.Department.findByPk(department) ?? {};
    const locationFilteredResults = locationGroup ? results.filter((result)=>result.locationHistory.some(({ to })=>{
            const { group, location } = _Location.Location.parseFullLocationName(to);
            return group ? group === locationGroupName : locationNames.includes(location);
        })) : results;
    const departmentFilteredResults = requiredDepartment ? locationFilteredResults.filter((result)=>result.departmentHistory.map(({ to })=>to).includes(requiredDepartment)) : locationFilteredResults;
    return departmentFilteredResults;
};
async function queryAdmissionsData(models, parameters) {
    const results = (await models.Encounter.findAll({
        include: [
            {
                model: models.Patient,
                as: 'patient',
                include: [
                    'village'
                ]
            },
            'examiner',
            'patientBillingType',
            {
                model: models.Location,
                as: 'location',
                include: [
                    'locationGroup'
                ]
            },
            'department',
            {
                model: models.EncounterDiagnosis,
                as: 'diagnoses',
                required: false,
                where: {
                    certainty: _constants.DIAGNOSIS_CERTAINTY.CONFIRMED
                },
                include: [
                    'Diagnosis'
                ]
            }
        ],
        where: parametersToSqlWhere(parameters)
    })).map((x)=>x.get({
            plain: true
        }));
    const encounterIds = results.map(({ id })=>id);
    const { locationChangeNotes, departmentChangeNotes } = await getAllNotes(models, encounterIds);
    const resultsWithHistory = results.map((result)=>({
            ...result,
            locationHistory: getPlaceHistoryFromNotes(locationChangeNotes, result, 'location'),
            departmentHistory: getPlaceHistoryFromNotes(departmentChangeNotes, result, 'department')
        }));
    const filteredResults = await filterResults(models, resultsWithHistory, parameters);
    return Promise.all(filteredResults.map(async (result)=>{
        const locationHistoryString = formatHistory(result.locationHistory, 'location');
        return {
            ...result,
            locationHistoryString,
            departmentHistoryString: formatHistory(result.departmentHistory, 'department'),
            primaryDiagnoses: stringifyDiagnoses(result.diagnoses, true),
            secondaryDiagnoses: stringifyDiagnoses(result.diagnoses, false)
        };
    }));
}
async function dataGenerator({ models }, parameters) {
    const queryResults = await queryAdmissionsData(models, parameters);
    return (0, _utilities.generateReportFromQueryData)(queryResults, reportColumnTemplate);
}

//# sourceMappingURL=admissions.js.map