"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "surveyResponseAnswer", {
    enumerable: true,
    get: function() {
        return surveyResponseAnswer;
    }
});
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _errors = require("@tamanu/shared/errors");
const _constants = require("@tamanu/constants");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const surveyResponseAnswer = _express.default.Router();
surveyResponseAnswer.put('/vital/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, user, params, settings, body: { facilityId, ...body } } = req;
    const { SurveyResponseAnswer, SurveyResponse, Survey, VitalLog, ProgramDataElement } = models;
    const { id } = params;
    const enableVitalEdit = await settings[facilityId].get('features.enableVitalEdit');
    if (!enableVitalEdit) {
        throw new _errors.InvalidOperationError('Editing vitals is disabled.');
    }
    req.checkPermission('write', 'Vitals');
    const answerObject = await SurveyResponseAnswer.findByPk(id, {
        include: [
            {
                required: true,
                model: SurveyResponse,
                as: 'surveyResponse',
                include: [
                    {
                        required: true,
                        model: Survey,
                        as: 'survey',
                        where: {
                            surveyType: _constants.SURVEY_TYPES.VITALS
                        }
                    }
                ]
            },
            {
                required: true,
                model: ProgramDataElement,
                where: {
                    type: {
                        [_sequelize.Op.not]: _constants.PROGRAM_DATA_ELEMENT_TYPES.CALCULATED
                    }
                }
            }
        ]
    });
    if (!answerObject) throw new _errors.NotFoundError();
    if (answerObject.body === body.newValue) {
        throw new _errors.InvalidParameterError('New value is the same as previous value.');
    }
    await db.transaction(async ()=>{
        const { newValue = '', reasonForChange, date } = body;
        await VitalLog.create({
            date,
            reasonForChange,
            previousValue: answerObject.body,
            newValue,
            recordedById: user.id,
            answerId: id
        });
        await answerObject.update({
            body: newValue
        });
        await answerObject.upsertCalculatedQuestions({
            date,
            reasonForChange,
            user
        });
    });
    res.send(answerObject);
}));
surveyResponseAnswer.post('/vital', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, user, settings, body: { facilityId, ...body } } = req;
    const { SurveyResponseAnswer, SurveyResponse, Survey, VitalLog, ProgramDataElement } = models;
    req.checkPermission('create', 'Vitals');
    // Even though this wouldn't technically be editing a vital
    // we will not allow the creation of a single vital answer if its not enabled
    const enableVitalEdit = await settings[facilityId].get('features.enableVitalEdit');
    if (!enableVitalEdit) {
        throw new _errors.InvalidOperationError('Editing vitals is disabled.');
    }
    // Ensure data element exists and it's not a calculated question
    const dataElement = await ProgramDataElement.findOne({
        where: {
            id: body.dataElementId
        }
    });
    if (!dataElement || dataElement.type === _constants.PROGRAM_DATA_ELEMENT_TYPES.CALCULATED) {
        throw new _errors.InvalidOperationError('Invalid data element.');
    }
    const responseObject = await SurveyResponse.findAll({
        where: {
            encounterId: body.encounterId
        },
        include: [
            {
                required: true,
                model: Survey,
                as: 'survey',
                where: {
                    surveyType: _constants.SURVEY_TYPES.VITALS
                }
            },
            {
                required: true,
                model: SurveyResponseAnswer,
                as: 'answers',
                where: {
                    body: body.recordedDate,
                    dataElementId: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded
                }
            }
        ]
    });
    // Can't do magic here, it's impossible to tell where
    // it should be created without guessing.
    if (responseObject.length !== 1) {
        throw new _errors.InvalidOperationError('Unable to complete action, please contact support.');
    }
    let newAnswer;
    await db.transaction(async ()=>{
        const { newValue = '', reasonForChange, date, dataElementId } = body;
        newAnswer = await models.SurveyResponseAnswer.create({
            dataElementId,
            body: newValue,
            responseId: responseObject[0].id
        });
        await VitalLog.create({
            date,
            reasonForChange,
            newValue,
            recordedById: user.id,
            answerId: newAnswer.id
        });
        await newAnswer.upsertCalculatedQuestions({
            date,
            reasonForChange,
            user
        });
    });
    res.send(newAnswer);
}));

//# sourceMappingURL=surveyResponseAnswer.js.map