"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "CentralServerConnection", {
    enumerable: true,
    get: function() {
        return CentralServerConnection;
    }
});
const _nodefetch = /*#__PURE__*/ _interop_require_default(require("node-fetch"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _errors = require("@tamanu/shared/errors");
const _constants = require("@tamanu/constants");
const _utils = require("@tamanu/shared/utils");
const _logging = require("@tamanu/shared/services/logging");
const _fetchWithTimeout = require("@tamanu/shared/utils/fetchWithTimeout");
const _sleepAsync = require("@tamanu/shared/utils/sleepAsync");
const _serverInfo = require("../serverInfo");
const _callWithBackoff = require("./callWithBackoff");
function _define_property(obj, key, value) {
    if (key in obj) {
        Object.defineProperty(obj, key, {
            value: value,
            enumerable: true,
            configurable: true,
            writable: true
        });
    } else {
        obj[key] = value;
    }
    return obj;
}
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const getVersionIncompatibleMessage = (error, response)=>{
    if (error.message === _constants.VERSION_COMPATIBILITY_ERRORS.LOW) {
        const minVersion = response.headers.get('X-Min-Client-Version');
        return `Please upgrade to Tamanu Facility Server v${minVersion} or higher.`;
    }
    if (error.message === _constants.VERSION_COMPATIBILITY_ERRORS.HIGH) {
        const maxVersion = response.headers.get('X-Max-Client-Version');
        return `The Tamanu Central Server only supports up to v${maxVersion} of the Facility Server, and needs to be upgraded. Please contact your system administrator.`;
    }
    return null;
};
const objectToQueryString = (obj)=>Object.entries(obj).filter(([k, v])=>k !== undefined && v !== undefined).map((kv)=>kv.map((str)=>encodeURIComponent(str)).join('=')).join('&');
let CentralServerConnection = class CentralServerConnection {
    async fetch(endpoint, params = {}) {
        const { headers = {}, body, method = 'GET', retryAuth = true, awaitConnection = true, backoff, ...otherParams } = params;
        // if there's an ongoing connection attempt, wait until it's finished
        // if we don't have a token, connect
        // allows deliberately skipping connect (so connect doesn't call itself)
        if (awaitConnection) {
            try {
                if (!this.token) {
                    // Deliberately use same backoff policy to avoid retrying in some places
                    await this.connect(backoff, otherParams.timeout);
                } else {
                    await this.connectionPromise;
                }
            } catch (e) {
            // ignore
            }
        }
        const url = `${this.host}/api/${endpoint}`;
        _logging.log.debug(`[sync] ${method} ${url}`);
        return (0, _callWithBackoff.callWithBackoff)(async ()=>{
            if (_config.default.debugging.requestFailureRate) {
                if (Math.random() < _config.default.debugging.requestFailureRate) {
                    // intended to cause some % of requests to fail, to simulate a flaky connection
                    throw new Error('Chaos: made your request fail');
                }
            }
            try {
                const response = await (0, _fetchWithTimeout.fetchWithTimeout)(url, {
                    method,
                    headers: {
                        Accept: 'application/json',
                        'X-Tamanu-Client': _constants.SERVER_TYPES.FACILITY,
                        'X-Version': _serverInfo.version,
                        Authorization: this.token ? `Bearer ${this.token}` : undefined,
                        'Content-Type': body ? 'application/json' : undefined,
                        ...headers
                    },
                    body: body && JSON.stringify(body),
                    timeout: this.timeout,
                    ...otherParams
                }, this.fetchImplementation);
                const isInvalidToken = response?.status === 401;
                if (isInvalidToken) {
                    if (retryAuth) {
                        _logging.log.warn('Token was invalid - reconnecting to central server');
                        await this.connect();
                        return this.fetch(endpoint, {
                            ...params,
                            retryAuth: false
                        });
                    }
                    throw new _errors.BadAuthenticationError(`Invalid credentials`);
                }
                if (!response.ok) {
                    const responseBody = await (0, _utils.getResponseJsonSafely)(response);
                    const { error } = responseBody;
                    // handle version incompatibility
                    if (response.status === 400 && error) {
                        const versionIncompatibleMessage = getVersionIncompatibleMessage(error, response);
                        if (versionIncompatibleMessage) {
                            throw new _errors.FacilityAndSyncVersionIncompatibleError(versionIncompatibleMessage);
                        }
                    }
                    const errorMessage = error ? error.message : 'no error message given';
                    const err = new _errors.RemoteCallFailedError(`Server responded with status code ${response.status} (${errorMessage})`);
                    // attach status and body from response
                    err.centralServerResponse = {
                        message: errorMessage,
                        status: response.status,
                        body: responseBody
                    };
                    throw err;
                }
                return await response.json();
            } catch (e) {
                // TODO: import AbortError from node-fetch once we're on v3.0
                if (e.name === 'AbortError') {
                    throw new _errors.RemoteTimeoutError(`Server failed to respond within ${this.timeout}ms - ${url}`);
                }
                throw e;
            }
        }, backoff);
    }
    async pollUntilTrue(endpoint) {
        const waitTime = 1000; // retry once per second
        const maxAttempts = 60 * 60 * 12; // for a maximum of 12 hours
        for(let attempt = 1; attempt <= maxAttempts; attempt++){
            const response = await this.fetch(endpoint);
            if (response) {
                return response;
            }
            await (0, _sleepAsync.sleepAsync)(waitTime);
        }
        throw new Error(`Did not get a truthy response after ${maxAttempts} attempts for ${endpoint}`);
    }
    async connect(backoff, timeout = this.timeout) {
        // if there's an ongoing connect attempt, reuse it
        if (this.connectionPromise) {
            return this.connectionPromise;
        }
        // store a promise for other functions to await
        this.connectionPromise = (async ()=>{
            const { email, password } = _config.default.sync;
            _logging.log.info(`Logging in to ${this.host} as ${email}...`);
            const facilityIds = (0, _utils.selectFacilityIds)(_config.default);
            const body = await this.fetch('login', {
                method: 'POST',
                body: {
                    email,
                    password,
                    facilityIds,
                    deviceId: this.deviceId
                },
                awaitConnection: false,
                retryAuth: false,
                backoff,
                timeout
            });
            if (!body.token || !body.user) {
                throw new _errors.BadAuthenticationError(`Encountered an unknown error while authenticating`);
            }
            _logging.log.info(`Received token for user ${body.user.displayName} (${body.user.email})`);
            this.token = body.token;
            return {
                ...body,
                serverFacilityIds: facilityIds
            };
        })();
        // await connection attempt, throwing an error if applicable, but always removing connectionPromise
        try {
            await this.connectionPromise;
            return this.connectionPromise;
        } finally{
            this.connectionPromise = null;
        }
    }
    async startSyncSession({ urgent, lastSyncedTick }) {
        const facilityIds = (0, _utils.selectFacilityIds)(_config.default);
        const { sessionId, status } = await this.fetch('sync', {
            method: 'POST',
            body: {
                facilityIds,
                deviceId: this.deviceId,
                urgent,
                lastSyncedTick
            }
        });
        if (!sessionId) {
            // we're waiting in a queue
            return {
                status
            };
        }
        // then, poll the sync/:sessionId/ready endpoint until we get a valid response
        // this is because POST /sync (especially the tickTockGlobalClock action) might get blocked
        // and take a while if the central server is concurrently persist records from another client
        await this.pollUntilTrue(`sync/${sessionId}/ready`);
        // finally, fetch the new tick from starting the session
        const { startedAtTick } = await this.fetch(`sync/${sessionId}/metadata`);
        return {
            sessionId,
            startedAtTick
        };
    }
    async endSyncSession(sessionId) {
        return this.fetch(`sync/${sessionId}`, {
            method: 'DELETE'
        });
    }
    async initiatePull(sessionId, since) {
        // first, set the pull filter on the central server, which will kick of a snapshot of changes
        // to pull
        const facilityIds = (0, _utils.selectFacilityIds)(_config.default);
        const body = {
            since,
            facilityIds,
            deviceId: this.deviceId
        };
        await this.fetch(`sync/${sessionId}/pull/initiate`, {
            method: 'POST',
            body
        });
        // then, poll the pull/ready endpoint until we get a valid response - it takes a while for
        // pull/initiate to finish populating the snapshot of changes
        await this.pollUntilTrue(`sync/${sessionId}/pull/ready`);
        // finally, fetch the metadata for the changes we're about to pull
        return this.fetch(`sync/${sessionId}/pull/metadata`);
    }
    async pull(sessionId, { limit = 100, fromId } = {}) {
        const query = {
            limit
        };
        if (fromId) {
            query.fromId = fromId;
        }
        const path = `sync/${sessionId}/pull?${objectToQueryString(query)}`;
        return this.fetch(path);
    }
    async push(sessionId, changes) {
        const path = `sync/${sessionId}/push`;
        return this.fetch(path, {
            method: 'POST',
            body: {
                changes
            }
        });
    }
    async completePush(sessionId) {
        // first off, mark the push as complete on central
        await this.fetch(`sync/${sessionId}/push/complete`, {
            method: 'POST',
            body: {
                deviceId: this.deviceId
            }
        });
        // now poll the complete check endpoint until we get a valid response - it takes a while for
        // the pushed changes to finish persisting to the central database
        await this.pollUntilTrue(`sync/${sessionId}/push/complete`);
    }
    async whoami() {
        return this.fetch('whoami');
    }
    async forwardRequest(req, endpoint) {
        try {
            const response = await this.fetch(endpoint, {
                method: req.method,
                body: req.body
            });
            return response;
        } catch (err) {
            if (err.centralServerResponse) {
                // pass central server response back
                const centralServerErrorMsg = err.centralServerResponse.body.error?.message;
                const passThroughError = new Error(centralServerErrorMsg ?? err);
                passThroughError.status = err.centralServerResponse.status;
                throw passThroughError;
            } else {
                // fallback
                throw new Error(`Central server error: ${err}`);
            }
        }
    }
    constructor({ deviceId }){
        _define_property(this, "connectionPromise", null);
        // test mocks don't always apply properly - this ensures the mock will be used
        _define_property(this, "fetchImplementation", _nodefetch.default);
        this.host = _config.default.sync.host.trim().replace(/\/*$/, '');
        this.timeout = _config.default.sync.timeout;
        this.batchSize = _config.default.sync.channelBatchSize;
        this.deviceId = deviceId;
    }
};

//# sourceMappingURL=CentralServerConnection.js.map