"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Appointment", {
    enumerable: true,
    get: function() {
        return Appointment;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _dateTimeTypes = require("./dateTimeTypes");
const _buildSyncLookupSelect = require("../sync/buildSyncLookupSelect");
let Appointment = class Appointment extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            startTime: (0, _dateTimeTypes.dateTimeType)('startTime', {
                allowNull: false
            }),
            endTime: (0, _dateTimeTypes.dateTimeType)('endTime'),
            type: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false,
                defaultValue: _constants.APPOINTMENT_TYPES.STANDARD
            },
            status: {
                type: _sequelize.Sequelize.STRING,
                allowNull: false,
                defaultValue: _constants.APPOINTMENT_STATUSES.CONFIRMED
            }
        }, {
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            ...options
        });
    }
    static getListReferenceAssociations() {
        return [
            {
                association: 'patient',
                include: [
                    'village'
                ]
            },
            'clinician',
            'location',
            'locationGroup'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            as: 'patient',
            foreignKey: 'patientId'
        });
        this.belongsTo(models.User, {
            as: 'clinician',
            foreignKey: 'clinicianId'
        });
        this.belongsTo(models.LocationGroup, {
            as: 'locationGroup',
            foreignKey: 'locationGroupId'
        });
        // Appointments are assigned a Location Group but the Location relation exists for legacy data
        this.belongsTo(models.Location, {
            as: 'location',
            foreignKey: 'locationId'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return `
      JOIN
        location_groups
      ON
        appointments.location_group_id = location_groups.id
      WHERE
        appointments.patient_id IN (SELECT patient_id FROM ${markedForSyncPatientsTable})
      AND
        location_groups.facility_id in (:facilityIds)
      AND
        appointments.updated_at_sync_tick > :since
    `;
    }
    static buildSyncLookupQueryDetails() {
        return {
            select: (0, _buildSyncLookupSelect.buildSyncLookupSelect)(this, {
                patientId: `${this.tableName}.patient_id`,
                facilityId: 'location_groups.facility_id'
            }),
            joins: `
        JOIN location_groups ON appointments.location_group_id = location_groups.id
      `
        };
    }
};

//# sourceMappingURL=Appointment.js.map