"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "PatientProgramRegistration", {
    enumerable: true,
    get: function() {
        return PatientProgramRegistration;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _dateTimeTypes = require("./dateTimeTypes");
const _dateTime = require("../utils/dateTime");
const _Model = require("./Model");
let PatientProgramRegistration = class PatientProgramRegistration extends _Model.Model {
    static init({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            date: (0, _dateTimeTypes.dateTimeType)('date', {
                allowNull: false,
                defaultValue: _dateTime.getCurrentDateTimeString
            }),
            registrationStatus: {
                allowNull: false,
                type: _sequelize.Sequelize.TEXT,
                defaultValue: _constants.REGISTRATION_STATUSES.ACTIVE
            },
            isMostRecent: {
                type: _sequelize.Sequelize.BOOLEAN,
                allowNull: false,
                defaultValue: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static getFullReferenceAssociations() {
        return [
            'programRegistry',
            'clinicalStatus',
            'clinician',
            'registeringFacility',
            'facility',
            'village'
        ];
    }
    static getListReferenceAssociations() {
        return [
            'clinicalStatus',
            'clinician'
        ];
    }
    static initRelations(models) {
        this.belongsTo(models.Patient, {
            foreignKey: {
                name: 'patientId',
                allowNull: false
            },
            as: 'patient'
        });
        this.belongsTo(models.ProgramRegistry, {
            foreignKey: {
                name: 'programRegistryId',
                allowNull: false
            },
            as: 'programRegistry'
        });
        this.belongsTo(models.ProgramRegistryClinicalStatus, {
            foreignKey: 'clinicalStatusId',
            as: 'clinicalStatus'
        });
        this.belongsTo(models.User, {
            foreignKey: {
                name: 'clinicianId',
                allowNull: false
            },
            as: 'clinician'
        });
        this.belongsTo(models.Facility, {
            foreignKey: 'registeringFacilityId',
            as: 'registeringFacility'
        });
        // 1. Note that only one of facilityId or villageId will usually be set,
        // depending on the currentlyAtType of the related programRegistry.
        this.belongsTo(models.Facility, {
            foreignKey: 'facilityId',
            as: 'facility'
        });
        this.belongsTo(models.ReferenceData, {
            foreignKey: 'villageId',
            as: 'village'
        });
    }
    static async create(values) {
        const { programRegistryId, patientId, ...restOfUpdates } = values;
        const existingRegistration = await this.sequelize.models.PatientProgramRegistration.findOne({
            where: {
                isMostRecent: true,
                programRegistryId,
                patientId
            }
        });
        // Most recent registration will now be the new one
        if (existingRegistration) {
            await existingRegistration.update({
                isMostRecent: false
            });
        }
        const newRegistrationValues = {
            patientId,
            programRegistryId,
            ...existingRegistration?.dataValues ?? {},
            // today's date should absolutely override the date of the previous registration record,
            // but if a date was provided in the function params, we should go with that.
            date: (0, _dateTime.getCurrentDateTimeString)(),
            ...restOfUpdates,
            isMostRecent: true
        };
        // Ensure a new id is generated, rather than using the one from existingRegistration
        delete newRegistrationValues.id;
        return super.create(newRegistrationValues);
    }
    static async getMostRecentRegistrationsForPatient(patientId) {
        return this.sequelize.models.PatientProgramRegistration.findAll({
            where: {
                isMostRecent: true,
                registrationStatus: {
                    [_sequelize.Op.ne]: _constants.REGISTRATION_STATUSES.RECORDED_IN_ERROR
                },
                patientId
            },
            include: [
                'clinicalStatus',
                'programRegistry'
            ],
            order: [
                // "active" > "removed"
                [
                    'registrationStatus',
                    'ASC'
                ],
                [
                    _sequelize.Sequelize.col('programRegistry.name'),
                    'ASC'
                ]
            ]
        });
    }
    static buildSyncFilter() {
        return null;
    }
    static buildSyncLookupQueryDetails() {
        return null; // syncs everywhere
    }
};

//# sourceMappingURL=PatientProgramRegistration.js.map