"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    constructPermission: function() {
        return constructPermission;
    },
    ensurePermissionCheck: function() {
        return ensurePermissionCheck;
    },
    getPermissions: function() {
        return getPermissions;
    },
    getSubjectName: function() {
        return getSubjectName;
    }
});
const _errors = require("../errors");
const _rolesToPermissions = require("./rolesToPermissions");
function getSubjectName(subject) {
    if (!subject || typeof subject === 'string') {
        return subject;
    }
    const Type = typeof subject === 'object' ? subject.constructor : subject;
    return Type.modelName || Type.name;
}
async function constructPermission(req, res, next) {
    try {
        // eslint-disable-next-line require-atomic-updates
        req.ability = await (0, _rolesToPermissions.getAbilityForUser)(req.models, req.user);
        next();
    } catch (e) {
        next(e);
    }
}
const checkPermission = (req, action, subject, field = '')=>{
    if (req.flagPermissionChecked) {
        req.flagPermissionChecked();
    }
    // allow a null permission to let things through - this means all endpoints
    // still need an explicit permission check, even if it's a null one!
    if (!action) {
        return;
    }
    const { ability } = req;
    if (!ability) {
        // user must log in - 401
        throw new _errors.BadAuthenticationError('No permission');
    }
    const subjectName = getSubjectName(subject);
    const hasPermission = ability.can(action, subject, field);
    if (req.audit) {
        req.audit.addPermissionCheck(action, subjectName, subject?.id);
    }
    if (!hasPermission) {
        // user is logged in fine, they're just not allowed - 403
        const rule = ability.relevantRuleFor(action, subject, field);
        const reason = rule && rule.reason || `No permission to perform action "${action}" on "${subjectName}"`;
        if (req.audit) {
            req.audit.annotate({
                forbiddenReason: reason
            });
        }
        throw new _errors.ForbiddenError(reason);
    }
};
function ensurePermissionCheck(req, res, next) {
    const originalResSend = res.send;
    req.checkPermission = (action, subject)=>{
        checkPermission(req, action, subject);
    };
    req.flagPermissionChecked = ()=>{
        res.send = originalResSend;
    };
    res.send = ()=>{
        res.send = originalResSend;
        res.status(501).send({
            error: {
                name: 'NoPermissionCheckError',
                message: 'No permission check was implemented for this endpoint.'
            }
        });
    };
    next();
}
async function getPermissions(req, res, _next) {
    const { user, models } = req;
    const permissions = await (0, _rolesToPermissions.getPermissionsForRoles)(models, user.role);
    res.send({
        permissions
    });
}

//# sourceMappingURL=middleware.js.map