import { Document, StyleSheet, View } from '@react-pdf/renderer';
import React from 'react';
import { CertificateHeader, styles } from './Layout';
import { PatientDetailsWithAddress } from './printComponents/PatientDetailsWithAddress';
import { DIAGNOSIS_CERTAINTIES_TO_HIDE } from '@tamanu/constants';
import { EncounterDetailsExtended } from './printComponents/EncounterDetailsExtended';
import { P } from './Typography';
import { LetterheadSection } from './LetterheadSection';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
const borderStyle = '1 solid black';
const tableLabelWidth = 150;
const tablePadding = 10;
const generalStyles = StyleSheet.create({
    tableContainer: {
        marginVertical: 5
    },
    sectionContainer: {
        marginVertical: 7
    }
});
const TableContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.tableContainer,
        ...props
    });
const SectionContainer = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: generalStyles.sectionContainer,
        ...props
    });
const infoBoxStyles = StyleSheet.create({
    row: {
        flexDirection: 'row',
        border: borderStyle
    },
    labelCol: {
        borderRight: borderStyle,
        width: tableLabelWidth,
        padding: tablePadding
    },
    dataCol: {
        flex: 1,
        padding: tablePadding
    },
    labelText: {
        fontFamily: 'Helvetica-Bold',
        fontSize: 10,
        fontWeight: 500
    },
    infoText: {
        fontFamily: 'Helvetica',
        fontSize: 10,
        fontWeight: 400
    }
});
const InfoBoxRow = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.row,
        ...props
    });
const InfoBoxLabelCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.labelCol,
        ...props
    });
const InfoBoxDataCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: infoBoxStyles.dataCol,
        ...props
    });
const medicationsTableStyles = StyleSheet.create({
    row: {
        flexDirection: 'row'
    },
    col: {
        flexDirection: 'column'
    },
    tableBorder: {
        border: borderStyle
    },
    titleCol: {
        width: tableLabelWidth,
        borderRight: borderStyle,
        padding: tablePadding
    },
    labelCol: {
        borderRight: borderStyle,
        width: 71,
        padding: tablePadding
    },
    dataCol: {
        flex: 1,
        padding: tablePadding
    }
});
const MedicationsTableBorder = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: [
            medicationsTableStyles.tableBorder,
            medicationsTableStyles.row
        ],
        ...props
    });
const MedicationsTableTitleCol = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: medicationsTableStyles.titleCol,
        ...props
    });
const notesSectionStyles = StyleSheet.create({
    notesBox: {
        border: borderStyle,
        minHeight: 76,
        padding: 10
    },
    title: {
        fontFamily: 'Helvetica-Bold',
        marginBottom: 3,
        fontSize: 11,
        fontWeight: 500
    }
});
const extractOngoingConditions = (patientConditions)=>patientConditions.map((item)=>item?.diagnosis?.name);
const extractDiagnosesInfo = ({ diagnoses, getSetting })=>{
    const displayIcd10Codes = getSetting('features.displayIcd10CodesInDischargeSummary');
    if (!displayIcd10Codes) {
        return diagnoses.map((item)=>item?.diagnosis?.name);
    } else {
        return diagnoses.map((item)=>`${item?.diagnosis?.name} (${item?.diagnosis?.code})`);
    }
};
const extractProceduresInfo = ({ procedures, getSetting })=>{
    const displayProcedureCodes = getSetting('features.displayProcedureCodesInDischargeSummary');
    if (!displayProcedureCodes) {
        return procedures.map((item)=>item?.procedureType?.name);
    } else {
        return procedures.map((item)=>`${item?.procedureType?.name} (${item?.procedureType?.code})`);
    }
};
const InfoBox = ({ label, info })=>/*#__PURE__*/ React.createElement(InfoBoxRow, null, /*#__PURE__*/ React.createElement(InfoBoxLabelCol, null, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.labelText
    }, label)), /*#__PURE__*/ React.createElement(InfoBoxDataCol, null, info.map((item, index)=>{
        return /*#__PURE__*/ React.createElement(Text, {
            style: infoBoxStyles.infoText,
            key: index
        }, item);
    })));
const DiagnosesTable = ({ title, diagnoses, getSetting })=>/*#__PURE__*/ React.createElement(InfoBox, {
        label: title,
        info: extractDiagnosesInfo({
            diagnoses,
            getSetting
        })
    });
const ProceduresTable = ({ procedures, getSetting })=>/*#__PURE__*/ React.createElement(InfoBox, {
        label: "Procedures",
        info: extractProceduresInfo({
            procedures,
            getSetting
        })
    });
const NotesSection = ({ notes })=>/*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(P, {
        bold: true,
        fontSize: 11,
        mb: 3
    }, "Discharge planning notes"), /*#__PURE__*/ React.createElement(View, {
        style: notesSectionStyles.notesBox
    }, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.infoText
    }, notes)));
const MedicationsTableInfoBox = ({ label, info, hasBottomBorder = false })=>/*#__PURE__*/ React.createElement(View, {
        style: {
            ...medicationsTableStyles.row,
            borderBottom: hasBottomBorder ?? borderStyle
        }
    }, /*#__PURE__*/ React.createElement(View, {
        style: medicationsTableStyles.labelCol
    }, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.infoText
    }, label)), /*#__PURE__*/ React.createElement(View, {
        style: medicationsTableStyles.dataCol
    }, info.map((item, index)=>{
        return /*#__PURE__*/ React.createElement(Text, {
            style: infoBoxStyles.infoText,
            key: index
        }, item);
    })));
const MedicationsTable = ({ medications })=>{
    const currentMedications = medications.filter((m)=>!m.discontinued);
    const discontinuedMedications = medications.filter((m)=>m.discontinued);
    return /*#__PURE__*/ React.createElement(MedicationsTableBorder, null, /*#__PURE__*/ React.createElement(MedicationsTableTitleCol, null, /*#__PURE__*/ React.createElement(Text, {
        style: infoBoxStyles.labelText
    }, "Medications")), /*#__PURE__*/ React.createElement(View, {
        style: {
            flexDirection: 'column',
            flex: 1
        }
    }, /*#__PURE__*/ React.createElement(MedicationsTableInfoBox, {
        label: "Current",
        info: currentMedications.map((item)=>item?.medication?.name),
        hasBottomBorder: true
    }), /*#__PURE__*/ React.createElement(MedicationsTableInfoBox, {
        label: "Discontinued",
        info: discontinuedMedications.map((item)=>item?.medication?.name),
        hasBottomBorder: false
    })));
};
const DischargeSummaryPrintoutComponent = ({ patientData, encounter, discharge, patientConditions, certificateData, getLocalisation, getTranslation, getSetting })=>{
    const { logo } = certificateData;
    const clinicianText = getTranslation('general.localisedField.clinician.label.short', 'Clinician');
    const { diagnoses, procedures, medications } = encounter;
    const visibleDiagnoses = diagnoses.filter(({ certainty })=>!DIAGNOSIS_CERTAINTIES_TO_HIDE.includes(certainty));
    const primaryDiagnoses = visibleDiagnoses.filter((d)=>d.isPrimary);
    const secondaryDiagnoses = visibleDiagnoses.filter((d)=>!d.isPrimary);
    const notes = discharge?.note;
    const { facilityName, facilityAddress, facilityTown } = discharge;
    // change header if facility details are present in discharge
    if (facilityName && facilityAddress && certificateData?.title) {
        certificateData = {
            ...certificateData,
            title: facilityName,
            subTitle: facilityTown ? `${facilityAddress}, ${facilityTown}` : facilityAddress
        };
    }
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: styles.page
    }, /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        certificateTitle: "Patient discharge summary",
        letterheadConfig: certificateData,
        logoSrc: logo
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(PatientDetailsWithAddress, {
        patient: patientData,
        getLocalisation: getLocalisation
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(EncounterDetailsExtended, {
        encounter: encounter,
        discharge: discharge,
        clinicianText: clinicianText
    })), /*#__PURE__*/ React.createElement(SectionContainer, null, patientConditions.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(InfoBox, {
        label: "Ongoing conditions",
        info: extractOngoingConditions(patientConditions)
    })), primaryDiagnoses.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(DiagnosesTable, {
        title: "Primary diagnoses",
        diagnoses: primaryDiagnoses,
        getSetting: getSetting
    })), secondaryDiagnoses.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(DiagnosesTable, {
        title: "Secondary diagnoses",
        diagnoses: secondaryDiagnoses,
        getSetting: getSetting
    })), procedures.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(ProceduresTable, {
        procedures: procedures,
        getSetting: getSetting
    })), medications.length > 0 && /*#__PURE__*/ React.createElement(TableContainer, null, /*#__PURE__*/ React.createElement(MedicationsTable, {
        medications: medications
    }))), /*#__PURE__*/ React.createElement(SectionContainer, null, /*#__PURE__*/ React.createElement(NotesSection, {
        notes: notes
    }))));
};
export const DischargeSummaryPrintout = withLanguageContext(DischargeSummaryPrintoutComponent);

//# sourceMappingURL=DischargeSummaryPrintout.js.map