import React from 'react';
import { Document, StyleSheet, View } from '@react-pdf/renderer';
import { PROGRAM_DATA_ELEMENT_TYPES } from '@tamanu/constants';
import { CertificateHeader, Watermark } from './Layout';
import { LetterheadSection } from './LetterheadSection';
import { MultiPageHeader } from './printComponents/MultiPageHeader';
import { getName } from '../patientAccessors';
import { Footer } from './printComponents/Footer';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
import { PatientDetails } from './printComponents/PatientDetails';
import { convertBinaryToYesNo, getResultName, getSurveyAnswerRows, separateColorText } from './surveyAnswers';
import { SurveyResponseDetails } from './printComponents/SurveyResponseDetails';
import { formatShort } from '../dateTime';
const pageStyles = StyleSheet.create({
    body: {
        paddingHorizontal: 50,
        paddingTop: 30,
        paddingBottom: 50
    },
    groupContainer: {
        display: 'flex',
        flexDirection: 'row',
        justifyContent: 'space-between',
        flexWrap: 'wrap',
        marginBottom: 4
    },
    item: {
        width: 238,
        display: 'flex',
        flexDirection: 'column',
        gap: 2,
        paddingBottom: 4,
        borderBottom: '0.5px solid black',
        marginBottom: 8,
        alignSelf: 'flex-end'
    },
    itemText: {
        fontSize: 9
    },
    boldText: {
        fontFamily: 'Helvetica-Bold'
    },
    boldDivider: {
        borderBottom: '2px solid black',
        height: 2,
        width: '100%',
        marginTop: '-6px'
    },
    resultBox: {
        paddingTop: 7,
        paddingLeft: 11,
        paddingBottom: 6,
        marginBottom: 14,
        fontSize: 11,
        border: '0.5px solid black',
        gap: 5
    }
});
const SectionSpacing = ({ height })=>/*#__PURE__*/ React.createElement(View, {
        style: {
            paddingBottom: height ?? '10px'
        }
    });
const ResultBox = ({ resultText, resultName })=>/*#__PURE__*/ React.createElement(View, {
        style: pageStyles.resultBox
    }, /*#__PURE__*/ React.createElement(Text, null, resultName), /*#__PURE__*/ React.createElement(Text, {
        style: [
            pageStyles.itemText,
            pageStyles.boldText
        ]
    }, resultText));
const getAnswers = ({ answer, type })=>{
    switch(type){
        case PROGRAM_DATA_ELEMENT_TYPES.RESULT:
            {
                const { strippedResultText } = separateColorText(answer);
                return strippedResultText;
            }
        case PROGRAM_DATA_ELEMENT_TYPES.CHECKBOX:
            return convertBinaryToYesNo(answer);
        case PROGRAM_DATA_ELEMENT_TYPES.CALCULATED:
            return parseFloat(answer).toFixed(1);
        case PROGRAM_DATA_ELEMENT_TYPES.PHOTO:
            return 'Image file - Refer to Tamanu to view';
        case PROGRAM_DATA_ELEMENT_TYPES.SUBMISSION_DATE:
            return formatShort(answer);
        case PROGRAM_DATA_ELEMENT_TYPES.DATE:
            return formatShort(answer);
        case PROGRAM_DATA_ELEMENT_TYPES.MULTI_SELECT:
            return JSON.parse(answer).join(', ');
        default:
            return answer;
    }
};
const ResponseItem = ({ row })=>{
    const { name, answer, type } = row;
    return /*#__PURE__*/ React.createElement(View, {
        style: pageStyles.item,
        wrap: false
    }, /*#__PURE__*/ React.createElement(Text, {
        style: pageStyles.itemText
    }, name), /*#__PURE__*/ React.createElement(Text, {
        style: [
            pageStyles.itemText,
            pageStyles.boldText
        ]
    }, getAnswers({
        answer,
        type
    })));
};
const ResponsesGroup = ({ rows })=>{
    return /*#__PURE__*/ React.createElement(View, {
        style: pageStyles.groupContainer
    }, rows.map((row)=>/*#__PURE__*/ React.createElement(ResponseItem, {
            key: row.id,
            row: row
        })), /*#__PURE__*/ React.createElement(View, {
        style: pageStyles.boldDivider
    }));
};
const SurveyResponsesPrintoutComponent = ({ patientData, certificateData, getLocalisation, surveyResponse, isReferral, facility, currentUser })=>{
    const { watermark, logo } = certificateData;
    const surveyAnswerRows = getSurveyAnswerRows(surveyResponse).filter(({ answer })=>answer);
    const groupedAnswerRows = Object.values(surveyAnswerRows.reduce((acc, item)=>{
        if (!acc[item.screenIndex]) {
            acc[item.screenIndex] = [];
        }
        acc[item.screenIndex].push(item);
        return acc;
    }, {}));
    const { strippedResultText } = separateColorText(surveyResponse.resultText);
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: pageStyles.body
    }, watermark && /*#__PURE__*/ React.createElement(Watermark, {
        src: watermark
    }), /*#__PURE__*/ React.createElement(MultiPageHeader, {
        documentName: !isReferral ? 'Program form' : 'Referral',
        documentSubname: surveyResponse.title,
        patientId: patientData.displayId,
        patientName: getName(patientData)
    }), /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        getLocalisation: getLocalisation,
        logoSrc: logo,
        certificateTitle: !isReferral ? 'Program form' : 'Referral',
        certificateSubtitle: surveyResponse.title,
        letterheadConfig: certificateData
    })), /*#__PURE__*/ React.createElement(SectionSpacing, null), /*#__PURE__*/ React.createElement(PatientDetails, {
        getLocalisation: getLocalisation,
        patient: patientData
    }), /*#__PURE__*/ React.createElement(SurveyResponseDetails, {
        surveyResponse: surveyResponse
    }), /*#__PURE__*/ React.createElement(SectionSpacing, {
        height: 16
    }), strippedResultText && /*#__PURE__*/ React.createElement(ResultBox, {
        resultText: strippedResultText,
        resultName: getResultName(surveyResponse.components)
    }), groupedAnswerRows.map((group, index)=>/*#__PURE__*/ React.createElement(ResponsesGroup, {
            key: index,
            rows: group
        })), /*#__PURE__*/ React.createElement(Footer, {
        printFacility: facility?.name,
        printedBy: currentUser?.displayName
    })));
};
export const SurveyResponsesPrintout = withLanguageContext(SurveyResponsesPrintoutComponent);

//# sourceMappingURL=SurveyResponsesPrintout.js.map